// SPDX-FileComment: This file is part of TNL - Template Numerical Library (https://tnl-project.org/)
// SPDX-License-Identifier: MIT

#include <filesystem>

#include <TNL/Config/parseCommandLine.h>
#include <TNL/Meshes/Mesh.h>
#include <TNL/Meshes/MeshOrdering.h>
#include <TNL/Meshes/MeshOrdering/CuthillMcKeeOrdering.h>
#include <TNL/Meshes/MeshOrdering/KdTreeOrdering.h>
#include <TNL/Meshes/MeshOrdering/HilbertOrdering.h>
#include <TNL/Meshes/TypeResolver/resolveMeshType.h>
#include <TNL/Meshes/Writers/VTKWriter.h>
#include <TNL/Meshes/Writers/VTUWriter.h>

using namespace TNL;
using namespace TNL::Meshes;

struct MyConfigTag
{};

namespace TNL::Meshes::BuildConfigTags {

/****
 * Turn off all grids.
 */
template<>
struct GridRealTag< MyConfigTag, float >
{
   static constexpr bool enabled = false;
};
template<>
struct GridRealTag< MyConfigTag, double >
{
   static constexpr bool enabled = false;
};
template<>
struct GridRealTag< MyConfigTag, long double >
{
   static constexpr bool enabled = false;
};

/****
 * Unstructured meshes.
 */
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Edge >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Triangle >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Quadrangle >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Polygon >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Tetrahedron >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Hexahedron >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Wedge >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Pyramid >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshCellTopologyTag< MyConfigTag, Topologies::Polyhedron >
{
   static constexpr bool enabled = true;
};

// Meshes are enabled only for the world dimension equal to the cell dimension.
template< typename CellTopology, int WorldDimension >
struct MeshSpaceDimensionTag< MyConfigTag, CellTopology, WorldDimension >
{
   static constexpr bool enabled = WorldDimension == CellTopology::dimension;
};

// Meshes are enabled only for types explicitly listed below.
template<>
struct MeshRealTag< MyConfigTag, float >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshRealTag< MyConfigTag, double >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshGlobalIndexTag< MyConfigTag, int >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshGlobalIndexTag< MyConfigTag, long int >
{
   static constexpr bool enabled = true;
};
template<>
struct MeshLocalIndexTag< MyConfigTag, short int >
{
   static constexpr bool enabled = true;
};

}  // namespace TNL::Meshes::BuildConfigTags

template< typename Mesh >
bool
reorder( Mesh&& mesh, const std::string& ordering, const std::string& outputFileName, std::string outputFileFormat )
{
   if( ordering == "rcm" ) {
      // reverse Cuthill-McKee ordering
      MeshOrdering< Mesh, CuthillMcKeeOrdering<> > rcm_ordering;
      rcm_ordering.reorder( mesh );
   }
   else if( ordering == "kdtree" ) {
#ifdef HAVE_CGAL
      // k-d tree ordering
      MeshOrdering< Mesh, KdTreeOrdering > kdtree_ordering;
      kdtree_ordering.reorder( mesh );
#else
      std::cerr << "CGAL support is missing. Recompile with -DHAVE_CGAL and try again.\n";
      return false;
#endif
   }
   else if( ordering == "hilbert" ) {
#ifdef HAVE_CGAL
      // Hilbert curve ordering
      MeshOrdering< Mesh, HilbertOrdering > hilbert_ordering;
      hilbert_ordering.reorder( mesh );
#else
      std::cerr << "CGAL support is missing. Recompile with -DHAVE_CGAL and try again.\n";
      return false;
#endif
   }
   else {
      std::cerr << "unknown ordering algorithm: '" << ordering << "'. Available options are: rcm, kdtree, hilbert.\n";
      return false;
   }

   namespace fs = std::filesystem;
   if( outputFileFormat == "auto" ) {
      outputFileFormat = fs::path( outputFileName ).extension().string();
      if( ! outputFileFormat.empty() )
         // remove dot from the extension
         outputFileFormat = outputFileFormat.substr( 1 );
   }

   if( outputFileFormat == "vtk" ) {
      std::ofstream file( outputFileName );
      using MeshWriter = TNL::Meshes::Writers::VTKWriter< Mesh >;
      MeshWriter writer( file );
      writer.template writeEntities< Mesh::getMeshDimension() >( mesh );
   }
   else if( outputFileFormat == "vtu" ) {
      std::ofstream file( outputFileName );
      using MeshWriter = TNL::Meshes::Writers::VTUWriter< Mesh >;
      MeshWriter writer( file );
      writer.template writeEntities< Mesh::getMeshDimension() >( mesh );
   }
   else {
      std::cerr << "unknown output file format: '" << outputFileFormat << "'. Available options are: vtk, vtu.\n";
      return false;
   }

   return true;
}

void
configSetup( Config::ConfigDescription& config )
{
   config.addDelimiter( "General settings:" );
   config.addRequiredEntry< std::string >( "input-mesh", "Input file with the distributed mesh." );
   config.addEntry< std::string >( "input-mesh-format", "Input mesh file format.", "auto" );
   config.addRequiredEntry< std::string >( "ordering", "Algorithm to reorder the mesh." );
   config.addEntryEnum( "rcm" );
   config.addEntryEnum( "kdtree" );
   config.addRequiredEntry< std::string >( "output-mesh", "Output file with the distributed mesh." );
   config.addEntry< std::string >( "output-mesh-format", "Output mesh file format.", "auto" );
}

int
main( int argc, char* argv[] )
{
   Config::ParameterContainer parameters;
   Config::ConfigDescription conf_desc;

   configSetup( conf_desc );

   if( ! parseCommandLine( argc, argv, conf_desc, parameters ) )
      return EXIT_FAILURE;

   const auto inputFileName = parameters.getParameter< std::string >( "input-mesh" );
   const auto inputFileFormat = parameters.getParameter< std::string >( "input-mesh-format" );
   const auto ordering = parameters.getParameter< std::string >( "ordering" );
   const auto outputFileName = parameters.getParameter< std::string >( "output-mesh" );
   const auto outputFileFormat = parameters.getParameter< std::string >( "output-mesh-format" );

   bool status = true;
   auto wrapper = [ & ]( auto& reader, auto&& mesh )
   {
      using MeshType = std::decay_t< decltype( mesh ) >;
      status = reorder( std::forward< MeshType >( mesh ), ordering, outputFileName, outputFileFormat );
   };
   try {
      resolveAndLoadMesh< MyConfigTag, Devices::Host >( wrapper, inputFileName, inputFileFormat );
   }
   catch( const std::exception& e ) {
      std::cerr << "Error: " << e.what() << '\n';
      return EXIT_FAILURE;
   }
   return static_cast< int >( ! status );
}
