// SPDX-FileComment: This file is part of TNL - Template Numerical Library (https://tnl-project.org/)
// SPDX-License-Identifier: MIT

#include <iostream>
#include <fstream>
#include <iomanip>
#include <vector>
#include <TNL/Algorithms/sort.h>
using namespace std;

#include "generators.h"
#include "Measurer.h"

#ifndef LOW_POW
   #define LOW_POW 10
#endif

#ifndef HIGH_POW
   #define HIGH_POW 25
#endif

#ifndef TRIES
   #define TRIES 20
#endif

using namespace TNL;
using namespace TNL::Algorithms;
using namespace TNL::Algorithms::Sorting;

template< typename Sorter >
void
start( ostream& out, const string& delim )
{
   out << "size" << delim;
   out << "random" << delim;
   out << "shuffle" << delim;
   out << "sorted" << delim;
   out << "almost" << delim;
   out << "decreasing" << delim;
   out << "gauss" << delim;
   out << "bucket" << delim;
   out << "stagger" << delim;
   out << "zero_entropy";
   out << '\n';

   int wrongAnsCnt = 0;

   for( int pow = LOW_POW; pow <= HIGH_POW; pow++ ) {
      int size = ( 1 << pow );
      vector< int > vec( size );

      out << "2^" << pow << delim << flush;
      out << fixed << setprecision( 3 );

      out << Measurer< Sorter >::measure( generateRandom( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateShuffle( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateSorted( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateAlmostSorted( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateDecreasing( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateGaussian( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateBucket( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateStaggered( size ), TRIES, wrongAnsCnt );
      out << delim << flush;

      out << Measurer< Sorter >::measure( generateZero_entropy( size ), TRIES, wrongAnsCnt );
      out << '\n';
   }

   if( wrongAnsCnt > 0 )
      std::cerr << wrongAnsCnt << "tries were sorted incorrectly\n";
}

int
main( int argc, char* argv[] )
{
   if( argc == 1 ) {
#ifdef __CUDACC__
      std::cout << "Quicksort on GPU ...\n";
      start< Quicksort >( cout, "\t" );
      std::cout << "Bitonic sort on GPU ...\n";
      start< BitonicSort >( cout, "\t" );

   // FIXME: clang 14 fails to compile the reference algorithms (e.g. due to compile errors in thrust or cub)
   #if defined( __CUDA__ ) && ! defined( __clang__ )
      #ifdef HAVE_CUDA_SAMPLES
      std::cout << "Manca quicksort on GPU ...\n";
      start< MancaQuicksort >( cout, "\t" );
      std::cout << "Nvidia bitonic sort on GPU ...\n";
      start< NvidiaBitonicSort >( cout, "\t" );
      #endif
      std::cout << "Cederman quicksort on GPU ...\n";
      start< CedermanQuicksort >( cout, "\t" );
      std::cout << "Thrust radixsort on GPU ...\n";
      start< ThrustRadixsort >( cout, "\t" );
   #endif
#endif

      std::cout << "STL sort on CPU ...\n";
      start< STLSort >( cout, "\t" );
   }
   else {
      std::ofstream out( argv[ 1 ] );
#ifdef __CUDACC__
      std::cout << "Quicksort on GPU ...\n";
      start< Quicksort >( out, "," );
      std::cout << "Bitonic sort on GPU ...\n";
      start< BitonicSort >( out, "," );

   // FIXME: clang 14 fails to compile the reference algorithms (e.g. due to compile errors in thrust or cub)
   #if defined( __CUDA__ ) && ! defined( __clang__ )
      #ifdef HAVE_CUDA_SAMPLES
      std::cout << "Manca quicksort on GPU ...\n";
      start< MancaQuicksort >( out, "," );
      std::cout << "Nvidia bitonic sort on GPU ...\n";
      start< NvidiaBitonicSort >( out, "," );
      #endif
      std::cout << "Cederman quicksort on GPU ...\n";
      start< CedermanQuicksort >( out, "," );
      std::cout << "Thrust radixsort on GPU ...\n";
      start< ThrustRadixsort >( out, "," );
   #endif
#endif

      std::cout << "STL sort on CPU ...\n";
      start< STLSort >( out, "," );
   }
   return 0;
}
