/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

use crate::derives::*;

type AtomThinVec = thin_vec::ThinVec<Atom>;

/// Gecko's pseudo-element definition.
///
/// We intentionally double-box legacy ::-moz-tree pseudo-elements to keep the
/// size of PseudoElement (and thus selector components) small.
#[derive(Clone, Debug, Eq, Hash, MallocSizeOf, PartialEq, ToShmem)]
#[repr(u8)]
pub enum PseudoElement {
    /// -moz-oof-placeholder
    MozOofPlaceholder,
    /// -moz-hframeset-border
    MozHframesetBorder,
    /// -moz-vframeset-border
    MozVframesetBorder,
    /// -moz-frameset-blank
    MozFramesetBlank,
    /// -moz-table-column-group
    MozTableColumnGroup,
    /// -moz-table-column
    MozTableColumn,
    /// -moz-page
    MozPage,
    /// -moz-page-break
    MozPageBreak,
    /// -moz-page-content
    MozPageContent,
    /// -moz-printed-sheet
    MozPrintedSheet,
    /// -moz-column-span-wrapper
    MozColumnSpanWrapper,
    /// -moz-text
    MozText,
    /// -moz-first-letter-continuation
    MozFirstLetterContinuation,
    /// -moz-block-inside-inline-wrapper
    MozBlockInsideInlineWrapper,
    /// -moz-mathml-anonymous-block
    MozMathmlAnonymousBlock,
    /// -moz-line-frame
    MozLineFrame,
    /// -moz-cell-content
    MozCellContent,
    /// -moz-fieldset-content
    MozFieldsetContent,
    /// -moz-html-canvas-content
    MozHtmlCanvasContent,
    /// -moz-inline-table
    MozInlineTable,
    /// -moz-table
    MozTable,
    /// -moz-table-cell
    MozTableCell,
    /// -moz-table-wrapper
    MozTableWrapper,
    /// -moz-table-row-group
    MozTableRowGroup,
    /// -moz-table-row
    MozTableRow,
    /// -moz-canvas
    MozCanvas,
    /// -moz-page-sequence
    MozPageSequence,
    /// -moz-scrolled-content
    MozScrolledContent,
    /// -moz-scrolled-canvas
    MozScrolledCanvas,
    /// -moz-column-set
    MozColumnSet,
    /// -moz-column-content
    MozColumnContent,
    /// -moz-viewport
    MozViewport,
    /// -moz-viewport-scroll
    MozViewportScroll,
    /// -moz-anonymous-item
    MozAnonymousItem,
    /// -moz-block-ruby-content
    MozBlockRubyContent,
    /// -moz-ruby
    MozRuby,
    /// -moz-ruby-base
    MozRubyBase,
    /// -moz-ruby-base-container
    MozRubyBaseContainer,
    /// -moz-ruby-text
    MozRubyText,
    /// -moz-ruby-text-container
    MozRubyTextContainer,
    /// -moz-svg-marker-anon-child
    MozSvgMarkerAnonChild,
    /// -moz-svg-outer-svg-anon-child
    MozSvgOuterSvgAnonChild,
    /// -moz-svg-foreign-content
    MozSvgForeignContent,
    /// -moz-svg-text
    MozSvgText,
    /// after
    After,
    /// before
    Before,
    /// marker
    Marker,
    /// backdrop
    Backdrop,
    /// cue
    Cue,
    /// first-letter
    FirstLetter,
    /// first-line
    FirstLine,
    /// highlight
    Highlight(AtomIdent),
    /// selection
    Selection,
    /// target-text
    TargetText,
    /// view-transition
    ViewTransition,
    /// view-transition-group
    ViewTransitionGroup(PtNameAndClassSelector),
    /// view-transition-image-pair
    ViewTransitionImagePair(PtNameAndClassSelector),
    /// view-transition-old
    ViewTransitionOld(PtNameAndClassSelector),
    /// view-transition-new
    ViewTransitionNew(PtNameAndClassSelector),
    /// -moz-snapshot-containing-block
    MozSnapshotContainingBlock,
    /// -moz-number-spin-box
    MozNumberSpinBox,
    /// -moz-number-spin-down
    MozNumberSpinDown,
    /// -moz-number-spin-up
    MozNumberSpinUp,
    /// -moz-search-clear-button
    MozSearchClearButton,
    /// -moz-select-content
    MozSelectContent,
    /// -moz-select-picker-icon
    MozSelectPickerIcon,
    /// -moz-progress-bar
    MozProgressBar,
    /// -moz-range-track
    MozRangeTrack,
    /// -moz-range-progress
    MozRangeProgress,
    /// -moz-range-thumb
    MozRangeThumb,
    /// -moz-meter-bar
    MozMeterBar,
    /// placeholder
    Placeholder,
    /// -moz-color-swatch
    MozColorSwatch,
    /// -moz-text-control-editing-root
    MozTextControlEditingRoot,
    /// -moz-text-control-preview
    MozTextControlPreview,
    /// -moz-reveal
    MozReveal,
    /// file-selector-button
    FileSelectorButton,
    /// -moz-file-content
    MozFileContent,
    /// slider-track
    SliderTrack,
    /// slider-thumb
    SliderThumb,
    /// slider-fill
    SliderFill,
    /// details-content
    DetailsContent,
    /// -moz-tree-column
    MozTreeColumn(AtomThinVec),
    /// -moz-tree-row
    MozTreeRow(AtomThinVec),
    /// -moz-tree-separator
    MozTreeSeparator(AtomThinVec),
    /// -moz-tree-cell
    MozTreeCell(AtomThinVec),
    /// -moz-tree-indentation
    MozTreeIndentation(AtomThinVec),
    /// -moz-tree-line
    MozTreeLine(AtomThinVec),
    /// -moz-tree-twisty
    MozTreeTwisty(AtomThinVec),
    /// -moz-tree-image
    MozTreeImage(AtomThinVec),
    /// -moz-tree-cell-text
    MozTreeCellText(AtomThinVec),
    /// -moz-tree-checkbox
    MozTreeCheckbox(AtomThinVec),
    /// -moz-tree-drop-feedback
    MozTreeDropFeedback(AtomThinVec),
    /// ::-webkit-* that we don't recognize
    /// https://github.com/whatwg/compat/issues/103
    UnknownWebkit(Atom),
}



/// The number of eager pseudo-elements.
pub const EAGER_PSEUDO_COUNT: usize = 4;

/// The number of all pseudo-elements.
pub const PSEUDO_COUNT: usize = 93;

/// The list of eager pseudos.
pub const EAGER_PSEUDOS: [PseudoElement; EAGER_PSEUDO_COUNT] = [
    PseudoElement::After,
    PseudoElement::Before,
    PseudoElement::FirstLetter,
    PseudoElement::FirstLine,
];



impl PseudoElement {
    /// Whether this pseudo-element is tree pseudo-element.
    #[inline]
    pub fn is_tree_pseudo_element(&self) -> bool {
        match *self {
            PseudoElement::MozTreeColumn(..) => true,
            PseudoElement::MozTreeRow(..) => true,
            PseudoElement::MozTreeSeparator(..) => true,
            PseudoElement::MozTreeCell(..) => true,
            PseudoElement::MozTreeIndentation(..) => true,
            PseudoElement::MozTreeLine(..) => true,
            PseudoElement::MozTreeTwisty(..) => true,
            PseudoElement::MozTreeImage(..) => true,
            PseudoElement::MozTreeCellText(..) => true,
            PseudoElement::MozTreeCheckbox(..) => true,
            PseudoElement::MozTreeDropFeedback(..) => true,
            _ => false,
        }
    }

    #[inline]
    fn flags_for_index(i: usize) -> PseudoStyleTypeFlags {
        static FLAGS: [PseudoStyleTypeFlags; PSEUDO_COUNT + 1] = [
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_NON_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits() | PseudoStyleTypeFlags::IS_WRAPPER_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_INHERITING_ANON_BOX.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_CSS2.bits() | PseudoStyleTypeFlags::IS_EAGER.bits() | PseudoStyleTypeFlags::IS_FLEX_OR_GRID_ITEM.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_CSS2.bits() | PseudoStyleTypeFlags::IS_EAGER.bits() | PseudoStyleTypeFlags::IS_FLEX_OR_GRID_ITEM.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_JS_CREATED_NAC.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_CSS2.bits() | PseudoStyleTypeFlags::IS_EAGER.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_CSS2.bits() | PseudoStyleTypeFlags::IS_EAGER.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::ENABLED_BY_PREF.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits() | PseudoStyleTypeFlags::IS_ELEMENT_BACKED.bits() | PseudoStyleTypeFlags::SUPPORTS_USER_ACTION_STATE.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::from_bits_truncate(PseudoStyleTypeFlags::ENABLED_IN_UA.bits() | PseudoStyleTypeFlags::ENABLED_IN_CHROME.bits() | PseudoStyleTypeFlags::IS_PSEUDO_ELEMENT.bits()),
        PseudoStyleTypeFlags::NONE, // :-webkit-*
        ];
        FLAGS[i]
    }

    /// Gets the flags associated to this pseudo-element or anon box.
    #[inline]
    pub fn flags(&self) -> PseudoStyleTypeFlags {
        Self::flags_for_index(self.index())
    }

    /// If this pseudo is pref-gated, returns the pref value, otherwise false.
    pub fn type_enabled_in_content(ty: PseudoStyleType) -> bool {
        let flags = Self::flags_for_index(ty as usize);
        // Common path, not enabled explicitly in UA sheets (note that chrome implies UA), and not
        // pref-gated.
        if !flags.intersects(PseudoStyleTypeFlags::ENABLED_IN_UA | PseudoStyleTypeFlags::ENABLED_BY_PREF) {
            return true;
        }
        if !flags.intersects(PseudoStyleTypeFlags::ENABLED_BY_PREF) {
            return false;
        }
        match ty {
            PseudoStyleType::ViewTransition => pref!("dom.viewTransitions.enabled"),
            PseudoStyleType::ViewTransitionGroup => pref!("dom.viewTransitions.enabled"),
            PseudoStyleType::ViewTransitionImagePair => pref!("dom.viewTransitions.enabled"),
            PseudoStyleType::ViewTransitionOld => pref!("dom.viewTransitions.enabled"),
            PseudoStyleType::ViewTransitionNew => pref!("dom.viewTransitions.enabled"),
            PseudoStyleType::SliderTrack => pref!("layout.css.modern-range-pseudos.enabled"),
            PseudoStyleType::SliderThumb => pref!("layout.css.modern-range-pseudos.enabled"),
            PseudoStyleType::SliderFill => pref!("layout.css.modern-range-pseudos.enabled"),
            PseudoStyleType::DetailsContent => pref!("layout.css.details-content.enabled"),
            _ => false,
        }
    }

    /// Construct a pseudo-element from a `PseudoStyleType`.
    #[inline]
    pub fn from_pseudo_type(type_: PseudoStyleType, functional_pseudo_parameter: Option<AtomIdent>) -> Option<Self> {
        match type_ {
            PseudoStyleType::MozOofPlaceholder => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozOofPlaceholder)
            },
            PseudoStyleType::MozHframesetBorder => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozHframesetBorder)
            },
            PseudoStyleType::MozVframesetBorder => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozVframesetBorder)
            },
            PseudoStyleType::MozFramesetBlank => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozFramesetBlank)
            },
            PseudoStyleType::MozTableColumnGroup => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableColumnGroup)
            },
            PseudoStyleType::MozTableColumn => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableColumn)
            },
            PseudoStyleType::MozPage => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozPage)
            },
            PseudoStyleType::MozPageBreak => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozPageBreak)
            },
            PseudoStyleType::MozPageContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozPageContent)
            },
            PseudoStyleType::MozPrintedSheet => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozPrintedSheet)
            },
            PseudoStyleType::MozColumnSpanWrapper => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozColumnSpanWrapper)
            },
            PseudoStyleType::MozText => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozText)
            },
            PseudoStyleType::MozFirstLetterContinuation => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozFirstLetterContinuation)
            },
            PseudoStyleType::MozBlockInsideInlineWrapper => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozBlockInsideInlineWrapper)
            },
            PseudoStyleType::MozMathmlAnonymousBlock => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozMathmlAnonymousBlock)
            },
            PseudoStyleType::MozLineFrame => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozLineFrame)
            },
            PseudoStyleType::MozCellContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozCellContent)
            },
            PseudoStyleType::MozFieldsetContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozFieldsetContent)
            },
            PseudoStyleType::MozHtmlCanvasContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozHtmlCanvasContent)
            },
            PseudoStyleType::MozInlineTable => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozInlineTable)
            },
            PseudoStyleType::MozTable => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTable)
            },
            PseudoStyleType::MozTableCell => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableCell)
            },
            PseudoStyleType::MozTableWrapper => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableWrapper)
            },
            PseudoStyleType::MozTableRowGroup => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableRowGroup)
            },
            PseudoStyleType::MozTableRow => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTableRow)
            },
            PseudoStyleType::MozCanvas => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozCanvas)
            },
            PseudoStyleType::MozPageSequence => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozPageSequence)
            },
            PseudoStyleType::MozScrolledContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozScrolledContent)
            },
            PseudoStyleType::MozScrolledCanvas => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozScrolledCanvas)
            },
            PseudoStyleType::MozColumnSet => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozColumnSet)
            },
            PseudoStyleType::MozColumnContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozColumnContent)
            },
            PseudoStyleType::MozViewport => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozViewport)
            },
            PseudoStyleType::MozViewportScroll => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozViewportScroll)
            },
            PseudoStyleType::MozAnonymousItem => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozAnonymousItem)
            },
            PseudoStyleType::MozBlockRubyContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozBlockRubyContent)
            },
            PseudoStyleType::MozRuby => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRuby)
            },
            PseudoStyleType::MozRubyBase => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRubyBase)
            },
            PseudoStyleType::MozRubyBaseContainer => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRubyBaseContainer)
            },
            PseudoStyleType::MozRubyText => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRubyText)
            },
            PseudoStyleType::MozRubyTextContainer => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRubyTextContainer)
            },
            PseudoStyleType::MozSvgMarkerAnonChild => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSvgMarkerAnonChild)
            },
            PseudoStyleType::MozSvgOuterSvgAnonChild => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSvgOuterSvgAnonChild)
            },
            PseudoStyleType::MozSvgForeignContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSvgForeignContent)
            },
            PseudoStyleType::MozSvgText => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSvgText)
            },
            PseudoStyleType::After => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::After)
            },
            PseudoStyleType::Before => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Before)
            },
            PseudoStyleType::Marker => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Marker)
            },
            PseudoStyleType::Backdrop => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Backdrop)
            },
            PseudoStyleType::Cue => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Cue)
            },
            PseudoStyleType::FirstLetter => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::FirstLetter)
            },
            PseudoStyleType::FirstLine => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::FirstLine)
            },
            
            PseudoStyleType::Highlight => {
                functional_pseudo_parameter.map(PseudoElement::Highlight)
            },
            PseudoStyleType::Selection => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Selection)
            },
            PseudoStyleType::TargetText => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::TargetText)
            },
            PseudoStyleType::ViewTransition => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::ViewTransition)
            },
            PseudoStyleType::ViewTransitionGroup => functional_pseudo_parameter.map(|p| {
                PseudoElement::ViewTransitionGroup(PtNameAndClassSelector::from_name(p.0))
            }),
            PseudoStyleType::ViewTransitionImagePair => functional_pseudo_parameter.map(|p| {
                PseudoElement::ViewTransitionImagePair(PtNameAndClassSelector::from_name(p.0))
            }),
            PseudoStyleType::ViewTransitionOld => functional_pseudo_parameter.map(|p| {
                PseudoElement::ViewTransitionOld(PtNameAndClassSelector::from_name(p.0))
            }),
            PseudoStyleType::ViewTransitionNew => functional_pseudo_parameter.map(|p| {
                PseudoElement::ViewTransitionNew(PtNameAndClassSelector::from_name(p.0))
            }),
            PseudoStyleType::MozSnapshotContainingBlock => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSnapshotContainingBlock)
            },
            PseudoStyleType::MozNumberSpinBox => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozNumberSpinBox)
            },
            PseudoStyleType::MozNumberSpinDown => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozNumberSpinDown)
            },
            PseudoStyleType::MozNumberSpinUp => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozNumberSpinUp)
            },
            PseudoStyleType::MozSearchClearButton => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSearchClearButton)
            },
            PseudoStyleType::MozSelectContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSelectContent)
            },
            PseudoStyleType::MozSelectPickerIcon => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozSelectPickerIcon)
            },
            PseudoStyleType::MozProgressBar => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozProgressBar)
            },
            PseudoStyleType::MozRangeTrack => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRangeTrack)
            },
            PseudoStyleType::MozRangeProgress => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRangeProgress)
            },
            PseudoStyleType::MozRangeThumb => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozRangeThumb)
            },
            PseudoStyleType::MozMeterBar => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozMeterBar)
            },
            PseudoStyleType::Placeholder => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::Placeholder)
            },
            PseudoStyleType::MozColorSwatch => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozColorSwatch)
            },
            PseudoStyleType::MozTextControlEditingRoot => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTextControlEditingRoot)
            },
            PseudoStyleType::MozTextControlPreview => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTextControlPreview)
            },
            PseudoStyleType::MozReveal => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozReveal)
            },
            PseudoStyleType::FileSelectorButton => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::FileSelectorButton)
            },
            PseudoStyleType::MozFileContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozFileContent)
            },
            PseudoStyleType::SliderTrack => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::SliderTrack)
            },
            PseudoStyleType::SliderThumb => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::SliderThumb)
            },
            PseudoStyleType::SliderFill => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::SliderFill)
            },
            PseudoStyleType::DetailsContent => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::DetailsContent)
            },
            PseudoStyleType::MozTreeColumn => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeColumn(Default::default()))
            },
            PseudoStyleType::MozTreeRow => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeRow(Default::default()))
            },
            PseudoStyleType::MozTreeSeparator => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeSeparator(Default::default()))
            },
            PseudoStyleType::MozTreeCell => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeCell(Default::default()))
            },
            PseudoStyleType::MozTreeIndentation => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeIndentation(Default::default()))
            },
            PseudoStyleType::MozTreeLine => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeLine(Default::default()))
            },
            PseudoStyleType::MozTreeTwisty => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeTwisty(Default::default()))
            },
            PseudoStyleType::MozTreeImage => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeImage(Default::default()))
            },
            PseudoStyleType::MozTreeCellText => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeCellText(Default::default()))
            },
            PseudoStyleType::MozTreeCheckbox => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeCheckbox(Default::default()))
            },
            PseudoStyleType::MozTreeDropFeedback => {
                debug_assert!(functional_pseudo_parameter.is_none());
                Some(PseudoElement::MozTreeDropFeedback(Default::default()))
            },
            _ => None,
        }
    }

    /// Construct a `PseudoStyleType`.
    #[inline]
    pub fn pseudo_type(&self) -> PseudoStyleType {
        // SAFETY: PseudoStyleType has the same variants as PseudoElement
        unsafe { std::mem::transmute::<u8, PseudoStyleType>(self.discriminant()) }
    }

    /// Returns the relevant PseudoStyleType, and an atom as an argument, if any.
    /// FIXME: we probably have to return the arguments of -moz-tree. However, they are multiple
    /// names, so we skip them for now (until we really need them).
    #[inline]
    pub fn pseudo_type_and_argument(&self) -> (PseudoStyleType, Option<&Atom>) {
        let ty = self.pseudo_type();
        let arg = match *self {
            PseudoElement::Highlight(ref val) => Some(&val.0),
            PseudoElement::ViewTransitionGroup(ref val) => Some(val.name()),
            PseudoElement::ViewTransitionImagePair(ref val) => Some(val.name()),
            PseudoElement::ViewTransitionOld(ref val) => Some(val.name()),
            PseudoElement::ViewTransitionNew(ref val) => Some(val.name()),
            _ => None,
        };
        (ty, arg)
    }

    /// Get the argument list of a tree pseudo-element.
    #[inline]
    pub fn tree_pseudo_args(&self) -> &[Atom] {
        match *self {
            PseudoElement::MozTreeColumn(ref args) => &args,
            PseudoElement::MozTreeRow(ref args) => &args,
            PseudoElement::MozTreeSeparator(ref args) => &args,
            PseudoElement::MozTreeCell(ref args) => &args,
            PseudoElement::MozTreeIndentation(ref args) => &args,
            PseudoElement::MozTreeLine(ref args) => &args,
            PseudoElement::MozTreeTwisty(ref args) => &args,
            PseudoElement::MozTreeImage(ref args) => &args,
            PseudoElement::MozTreeCellText(ref args) => &args,
            PseudoElement::MozTreeCheckbox(ref args) => &args,
            PseudoElement::MozTreeDropFeedback(ref args) => &args,
            _ => &[],
        }
    }

    /// Constructs a pseudo-element from a string of text.
    ///
    /// Returns `None` if the pseudo-element is not recognised.
    #[inline]
    pub fn from_slice(name: &str, allow_unkown_webkit: bool) -> Option<Self> {
        // We don't need to support tree pseudos because functional
        // pseudo-elements needs arguments, and thus should be created
        // via other methods.
        cssparser::ascii_case_insensitive_phf_map! {
            pseudo -> PseudoElement = {
                "-moz-oof-placeholder" => PseudoElement::MozOofPlaceholder,
                "-moz-hframeset-border" => PseudoElement::MozHframesetBorder,
                "-moz-vframeset-border" => PseudoElement::MozVframesetBorder,
                "-moz-frameset-blank" => PseudoElement::MozFramesetBlank,
                "-moz-table-column-group" => PseudoElement::MozTableColumnGroup,
                "-moz-table-column" => PseudoElement::MozTableColumn,
                "-moz-page" => PseudoElement::MozPage,
                "-moz-page-break" => PseudoElement::MozPageBreak,
                "-moz-page-content" => PseudoElement::MozPageContent,
                "-moz-printed-sheet" => PseudoElement::MozPrintedSheet,
                "-moz-column-span-wrapper" => PseudoElement::MozColumnSpanWrapper,
                "-moz-text" => PseudoElement::MozText,
                "-moz-first-letter-continuation" => PseudoElement::MozFirstLetterContinuation,
                "-moz-block-inside-inline-wrapper" => PseudoElement::MozBlockInsideInlineWrapper,
                "-moz-mathml-anonymous-block" => PseudoElement::MozMathmlAnonymousBlock,
                "-moz-line-frame" => PseudoElement::MozLineFrame,
                "-moz-cell-content" => PseudoElement::MozCellContent,
                "-moz-fieldset-content" => PseudoElement::MozFieldsetContent,
                "-moz-html-canvas-content" => PseudoElement::MozHtmlCanvasContent,
                "-moz-inline-table" => PseudoElement::MozInlineTable,
                "-moz-table" => PseudoElement::MozTable,
                "-moz-table-cell" => PseudoElement::MozTableCell,
                "-moz-table-wrapper" => PseudoElement::MozTableWrapper,
                "-moz-table-row-group" => PseudoElement::MozTableRowGroup,
                "-moz-table-row" => PseudoElement::MozTableRow,
                "-moz-canvas" => PseudoElement::MozCanvas,
                "-moz-page-sequence" => PseudoElement::MozPageSequence,
                "-moz-scrolled-content" => PseudoElement::MozScrolledContent,
                "-moz-scrolled-canvas" => PseudoElement::MozScrolledCanvas,
                "-moz-column-set" => PseudoElement::MozColumnSet,
                "-moz-column-content" => PseudoElement::MozColumnContent,
                "-moz-viewport" => PseudoElement::MozViewport,
                "-moz-viewport-scroll" => PseudoElement::MozViewportScroll,
                "-moz-anonymous-item" => PseudoElement::MozAnonymousItem,
                "-moz-block-ruby-content" => PseudoElement::MozBlockRubyContent,
                "-moz-ruby" => PseudoElement::MozRuby,
                "-moz-ruby-base" => PseudoElement::MozRubyBase,
                "-moz-ruby-base-container" => PseudoElement::MozRubyBaseContainer,
                "-moz-ruby-text" => PseudoElement::MozRubyText,
                "-moz-ruby-text-container" => PseudoElement::MozRubyTextContainer,
                "-moz-svg-marker-anon-child" => PseudoElement::MozSvgMarkerAnonChild,
                "-moz-svg-outer-svg-anon-child" => PseudoElement::MozSvgOuterSvgAnonChild,
                "-moz-svg-foreign-content" => PseudoElement::MozSvgForeignContent,
                "-moz-svg-text" => PseudoElement::MozSvgText,
                "after" => PseudoElement::After,
                "before" => PseudoElement::Before,
                "marker" => PseudoElement::Marker,
                "backdrop" => PseudoElement::Backdrop,
                "cue" => PseudoElement::Cue,
                "first-letter" => PseudoElement::FirstLetter,
                "first-line" => PseudoElement::FirstLine,
                "selection" => PseudoElement::Selection,
                "target-text" => PseudoElement::TargetText,
                "view-transition" => PseudoElement::ViewTransition,
                "-moz-snapshot-containing-block" => PseudoElement::MozSnapshotContainingBlock,
                "-moz-number-spin-box" => PseudoElement::MozNumberSpinBox,
                "-moz-number-spin-down" => PseudoElement::MozNumberSpinDown,
                "-moz-number-spin-up" => PseudoElement::MozNumberSpinUp,
                "-moz-search-clear-button" => PseudoElement::MozSearchClearButton,
                "-moz-select-content" => PseudoElement::MozSelectContent,
                "-moz-select-picker-icon" => PseudoElement::MozSelectPickerIcon,
                "-moz-progress-bar" => PseudoElement::MozProgressBar,
                "-moz-range-track" => PseudoElement::MozRangeTrack,
                "-moz-range-progress" => PseudoElement::MozRangeProgress,
                "-moz-range-thumb" => PseudoElement::MozRangeThumb,
                "-moz-meter-bar" => PseudoElement::MozMeterBar,
                "placeholder" => PseudoElement::Placeholder,
                "-moz-color-swatch" => PseudoElement::MozColorSwatch,
                "-moz-text-control-editing-root" => PseudoElement::MozTextControlEditingRoot,
                "-moz-text-control-preview" => PseudoElement::MozTextControlPreview,
                "-moz-reveal" => PseudoElement::MozReveal,
                "file-selector-button" => PseudoElement::FileSelectorButton,
                "-moz-file-content" => PseudoElement::MozFileContent,
                "slider-track" => PseudoElement::SliderTrack,
                "slider-thumb" => PseudoElement::SliderThumb,
                "slider-fill" => PseudoElement::SliderFill,
                "details-content" => PseudoElement::DetailsContent,
                // Alias some legacy prefixed pseudos to their standardized name at parse time:
                "-moz-selection" => PseudoElement::Selection,
                "-moz-placeholder" => PseudoElement::Placeholder,
                "-moz-list-bullet" => PseudoElement::Marker,
                "-moz-list-number" => PseudoElement::Marker,
            }
        }
        if let Some(p) = pseudo::get(name) {
            return Some(p.clone());
        }
        if starts_with_ignore_ascii_case(name, "-moz-tree-") {
            return PseudoElement::tree_pseudo_element(name, Default::default())
        }
        const WEBKIT_PREFIX: &str = "-webkit-";
        if allow_unkown_webkit && starts_with_ignore_ascii_case(name, WEBKIT_PREFIX) {
            let part = string_as_ascii_lowercase(&name[WEBKIT_PREFIX.len()..]);
            return Some(PseudoElement::UnknownWebkit(part.into()));
        }
        None
    }

    /// Constructs a tree pseudo-element from the given name and arguments.
    /// "name" must start with "-moz-tree-".
    ///
    /// Returns `None` if the pseudo-element is not recognized.
    #[inline]
    pub fn tree_pseudo_element(name: &str, args: thin_vec::ThinVec<Atom>) -> Option<Self> {
        debug_assert!(starts_with_ignore_ascii_case(name, "-moz-tree-"));
        let tree_part = &name[10..];
        if tree_part.eq_ignore_ascii_case("column") {
            return Some(PseudoElement::MozTreeColumn(args));
        }
        if tree_part.eq_ignore_ascii_case("row") {
            return Some(PseudoElement::MozTreeRow(args));
        }
        if tree_part.eq_ignore_ascii_case("separator") {
            return Some(PseudoElement::MozTreeSeparator(args));
        }
        if tree_part.eq_ignore_ascii_case("cell") {
            return Some(PseudoElement::MozTreeCell(args));
        }
        if tree_part.eq_ignore_ascii_case("indentation") {
            return Some(PseudoElement::MozTreeIndentation(args));
        }
        if tree_part.eq_ignore_ascii_case("line") {
            return Some(PseudoElement::MozTreeLine(args));
        }
        if tree_part.eq_ignore_ascii_case("twisty") {
            return Some(PseudoElement::MozTreeTwisty(args));
        }
        if tree_part.eq_ignore_ascii_case("image") {
            return Some(PseudoElement::MozTreeImage(args));
        }
        if tree_part.eq_ignore_ascii_case("cell-text") {
            return Some(PseudoElement::MozTreeCellText(args));
        }
        if tree_part.eq_ignore_ascii_case("checkbox") {
            return Some(PseudoElement::MozTreeCheckbox(args));
        }
        if tree_part.eq_ignore_ascii_case("drop-feedback") {
            return Some(PseudoElement::MozTreeDropFeedback(args));
        }
        None
    }

    /// Returns true if this pseudo-element matches the given selector.
    pub fn matches(
        &self,
        pseudo_selector: &PseudoElement,
        element: &super::wrapper::GeckoElement,
    ) -> bool {
        if *self == *pseudo_selector {
            return true;
        }

        if std::mem::discriminant(self) != std::mem::discriminant(pseudo_selector) {
            return false;
        }

        // Check named view transition pseudo-elements.
        self.matches_named_view_transition_pseudo_element(pseudo_selector, element)
    }
}

impl ToCss for PseudoElement {
    fn to_css<W>(&self, dest: &mut W) -> fmt::Result where W: fmt::Write {
        match *self {
                PseudoElement::MozOofPlaceholder => dest.write_str("::-moz-oof-placeholder"),
                PseudoElement::MozHframesetBorder => dest.write_str("::-moz-hframeset-border"),
                PseudoElement::MozVframesetBorder => dest.write_str("::-moz-vframeset-border"),
                PseudoElement::MozFramesetBlank => dest.write_str("::-moz-frameset-blank"),
                PseudoElement::MozTableColumnGroup => dest.write_str("::-moz-table-column-group"),
                PseudoElement::MozTableColumn => dest.write_str("::-moz-table-column"),
                PseudoElement::MozPage => dest.write_str("::-moz-page"),
                PseudoElement::MozPageBreak => dest.write_str("::-moz-page-break"),
                PseudoElement::MozPageContent => dest.write_str("::-moz-page-content"),
                PseudoElement::MozPrintedSheet => dest.write_str("::-moz-printed-sheet"),
                PseudoElement::MozColumnSpanWrapper => dest.write_str("::-moz-column-span-wrapper"),
                PseudoElement::MozText => dest.write_str("::-moz-text"),
                PseudoElement::MozFirstLetterContinuation => dest.write_str("::-moz-first-letter-continuation"),
                PseudoElement::MozBlockInsideInlineWrapper => dest.write_str("::-moz-block-inside-inline-wrapper"),
                PseudoElement::MozMathmlAnonymousBlock => dest.write_str("::-moz-mathml-anonymous-block"),
                PseudoElement::MozLineFrame => dest.write_str("::-moz-line-frame"),
                PseudoElement::MozCellContent => dest.write_str("::-moz-cell-content"),
                PseudoElement::MozFieldsetContent => dest.write_str("::-moz-fieldset-content"),
                PseudoElement::MozHtmlCanvasContent => dest.write_str("::-moz-html-canvas-content"),
                PseudoElement::MozInlineTable => dest.write_str("::-moz-inline-table"),
                PseudoElement::MozTable => dest.write_str("::-moz-table"),
                PseudoElement::MozTableCell => dest.write_str("::-moz-table-cell"),
                PseudoElement::MozTableWrapper => dest.write_str("::-moz-table-wrapper"),
                PseudoElement::MozTableRowGroup => dest.write_str("::-moz-table-row-group"),
                PseudoElement::MozTableRow => dest.write_str("::-moz-table-row"),
                PseudoElement::MozCanvas => dest.write_str("::-moz-canvas"),
                PseudoElement::MozPageSequence => dest.write_str("::-moz-page-sequence"),
                PseudoElement::MozScrolledContent => dest.write_str("::-moz-scrolled-content"),
                PseudoElement::MozScrolledCanvas => dest.write_str("::-moz-scrolled-canvas"),
                PseudoElement::MozColumnSet => dest.write_str("::-moz-column-set"),
                PseudoElement::MozColumnContent => dest.write_str("::-moz-column-content"),
                PseudoElement::MozViewport => dest.write_str("::-moz-viewport"),
                PseudoElement::MozViewportScroll => dest.write_str("::-moz-viewport-scroll"),
                PseudoElement::MozAnonymousItem => dest.write_str("::-moz-anonymous-item"),
                PseudoElement::MozBlockRubyContent => dest.write_str("::-moz-block-ruby-content"),
                PseudoElement::MozRuby => dest.write_str("::-moz-ruby"),
                PseudoElement::MozRubyBase => dest.write_str("::-moz-ruby-base"),
                PseudoElement::MozRubyBaseContainer => dest.write_str("::-moz-ruby-base-container"),
                PseudoElement::MozRubyText => dest.write_str("::-moz-ruby-text"),
                PseudoElement::MozRubyTextContainer => dest.write_str("::-moz-ruby-text-container"),
                PseudoElement::MozSvgMarkerAnonChild => dest.write_str("::-moz-svg-marker-anon-child"),
                PseudoElement::MozSvgOuterSvgAnonChild => dest.write_str("::-moz-svg-outer-svg-anon-child"),
                PseudoElement::MozSvgForeignContent => dest.write_str("::-moz-svg-foreign-content"),
                PseudoElement::MozSvgText => dest.write_str("::-moz-svg-text"),
                PseudoElement::After => dest.write_str("::after"),
                PseudoElement::Before => dest.write_str("::before"),
                PseudoElement::Marker => dest.write_str("::marker"),
                PseudoElement::Backdrop => dest.write_str("::backdrop"),
                PseudoElement::Cue => dest.write_str("::cue"),
                PseudoElement::FirstLetter => dest.write_str("::first-letter"),
                PseudoElement::FirstLine => dest.write_str("::first-line"),
                PseudoElement::Highlight(ref arg) => {
                    dest.write_str("::highlight(")?;
                    arg.to_css(dest)?;
                    dest.write_char(')')
                }
                PseudoElement::Selection => dest.write_str("::selection"),
                PseudoElement::TargetText => dest.write_str("::target-text"),
                PseudoElement::ViewTransition => dest.write_str("::view-transition"),
                PseudoElement::ViewTransitionGroup(ref arg) => {
                    dest.write_str("::view-transition-group(")?;
                    arg.to_css(dest)?;
                    dest.write_char(')')
                }
                PseudoElement::ViewTransitionImagePair(ref arg) => {
                    dest.write_str("::view-transition-image-pair(")?;
                    arg.to_css(dest)?;
                    dest.write_char(')')
                }
                PseudoElement::ViewTransitionOld(ref arg) => {
                    dest.write_str("::view-transition-old(")?;
                    arg.to_css(dest)?;
                    dest.write_char(')')
                }
                PseudoElement::ViewTransitionNew(ref arg) => {
                    dest.write_str("::view-transition-new(")?;
                    arg.to_css(dest)?;
                    dest.write_char(')')
                }
                PseudoElement::MozSnapshotContainingBlock => dest.write_str("::-moz-snapshot-containing-block"),
                PseudoElement::MozNumberSpinBox => dest.write_str("::-moz-number-spin-box"),
                PseudoElement::MozNumberSpinDown => dest.write_str("::-moz-number-spin-down"),
                PseudoElement::MozNumberSpinUp => dest.write_str("::-moz-number-spin-up"),
                PseudoElement::MozSearchClearButton => dest.write_str("::-moz-search-clear-button"),
                PseudoElement::MozSelectContent => dest.write_str("::-moz-select-content"),
                PseudoElement::MozSelectPickerIcon => dest.write_str("::-moz-select-picker-icon"),
                PseudoElement::MozProgressBar => dest.write_str("::-moz-progress-bar"),
                PseudoElement::MozRangeTrack => dest.write_str("::-moz-range-track"),
                PseudoElement::MozRangeProgress => dest.write_str("::-moz-range-progress"),
                PseudoElement::MozRangeThumb => dest.write_str("::-moz-range-thumb"),
                PseudoElement::MozMeterBar => dest.write_str("::-moz-meter-bar"),
                PseudoElement::Placeholder => dest.write_str("::placeholder"),
                PseudoElement::MozColorSwatch => dest.write_str("::-moz-color-swatch"),
                PseudoElement::MozTextControlEditingRoot => dest.write_str("::-moz-text-control-editing-root"),
                PseudoElement::MozTextControlPreview => dest.write_str("::-moz-text-control-preview"),
                PseudoElement::MozReveal => dest.write_str("::-moz-reveal"),
                PseudoElement::FileSelectorButton => dest.write_str("::file-selector-button"),
                PseudoElement::MozFileContent => dest.write_str("::-moz-file-content"),
                PseudoElement::SliderTrack => dest.write_str("::slider-track"),
                PseudoElement::SliderThumb => dest.write_str("::slider-thumb"),
                PseudoElement::SliderFill => dest.write_str("::slider-fill"),
                PseudoElement::DetailsContent => dest.write_str("::details-content"),
            PseudoElement::MozTreeColumn(ref args) => {
                dest.write_str("::-moz-tree-column")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeRow(ref args) => {
                dest.write_str("::-moz-tree-row")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeSeparator(ref args) => {
                dest.write_str("::-moz-tree-separator")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeCell(ref args) => {
                dest.write_str("::-moz-tree-cell")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeIndentation(ref args) => {
                dest.write_str("::-moz-tree-indentation")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeLine(ref args) => {
                dest.write_str("::-moz-tree-line")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeTwisty(ref args) => {
                dest.write_str("::-moz-tree-twisty")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeImage(ref args) => {
                dest.write_str("::-moz-tree-image")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeCellText(ref args) => {
                dest.write_str("::-moz-tree-cell-text")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeCheckbox(ref args) => {
                dest.write_str("::-moz-tree-checkbox")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::MozTreeDropFeedback(ref args) => {
                dest.write_str("::-moz-tree-drop-feedback")?;
                let mut iter = args.iter();
                if let Some(first) = iter.next() {
                    dest.write_char('(')?;
                    serialize_atom_identifier(&first, dest)?;
                    for item in iter {
                        dest.write_str(", ")?;
                        serialize_atom_identifier(item, dest)?;
                    }
                    dest.write_char(')')?;
                }
                Ok(())
            },
            PseudoElement::UnknownWebkit(ref atom) => {
                dest.write_str("::-webkit-")?;
                serialize_atom_identifier(atom, dest)
            },
        }
    }
}
