/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef _nsWifiAccessPoint_
#define _nsWifiAccessPoint_

#include <algorithm>
#include "nsWifiMonitor.h"
#include "nsIWifiAccessPoint.h"

#include "nsString.h"
#include "nsCOMArray.h"
#include "mozilla/Sprintf.h"

class nsWifiAccessPoint final : public nsIWifiAccessPoint {
  ~nsWifiAccessPoint() = default;

 public:
  NS_DECL_THREADSAFE_ISUPPORTS
  NS_DECL_NSIWIFIACCESSPOINT

  nsWifiAccessPoint();

  char mMac[18]{0};
  int mSignal;
  char mSsid[33]{0};
  int mSsidLen;

  void setSignal(int signal) { mSignal = signal; }

  void setMacRaw(const char* aString) {
    memcpy(mMac, aString, std::size(mMac));
  }

  void setMac(const unsigned char mac_as_int[6]) {
    // mac_as_int is big-endian. Write in byte chunks.
    // Format is XX-XX-XX-XX-XX-XX.

    const unsigned char holder[6] = {0};
    if (!mac_as_int) {
      mac_as_int = holder;
    }

    static const char* kMacFormatString = ("%02x-%02x-%02x-%02x-%02x-%02x");

    SprintfLiteral(mMac, kMacFormatString, mac_as_int[0], mac_as_int[1],
                   mac_as_int[2], mac_as_int[3], mac_as_int[4], mac_as_int[5]);

    mMac[17] = 0;
  }

  void setSSIDRaw(const char* aSSID, size_t len) {
    mSsidLen = std::min(len, std::size(mSsid));
    memcpy(mSsid, aSSID, mSsidLen);
  }

  void setSSID(const char* aSSID, unsigned long len) {
    if (aSSID && (len < sizeof(mSsid))) {
      strncpy(mSsid, aSSID, len);
      mSsid[len] = 0;
      mSsidLen = len;
    } else {
      mSsid[0] = 0;
      mSsidLen = 0;
    }
  }

  // 3-value compare for nsWifiAccessPoint
  int Compare(const nsWifiAccessPoint& o) const;

  bool operator==(const nsWifiAccessPoint& o) const;
  bool operator!=(const nsWifiAccessPoint& o) const { return !(*this == o); }
};

#endif
