/**
   r_memory.c


   Copyright (C) 2004, Network Resonance, Inc.
   Copyright (C) 2006, Network Resonance, Inc.
   All Rights Reserved

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. Neither the name of Network Resonance, Inc. nor the name of any
      contributors to this software may be used to endorse or promote
      products derived from this software without specific prior written
      permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE.


   ekr@rtfm.com  Thu Apr 22 20:40:45 2004
 */

#include <string.h>
#include <stddef.h>
#include <assert.h>
#include "r_common.h"
#include "r_memory.h"

#ifndef NO_MALLOC_REPLACE

typedef struct r_malloc_chunk_ {
#ifdef SANITY_CHECKS
     UINT4 hdr;
#endif
     UCHAR type;
     UINT4 size;
     UCHAR memory[1];
} r_malloc_chunk;

#define CHUNK_MEMORY_OFFSET                    offsetof(struct r_malloc_chunk_, memory)
#define GET_CHUNK_ADDR_FROM_MEM_ADDR(memp) \
        ((r_malloc_chunk *)(((unsigned char*)(memp))-CHUNK_MEMORY_OFFSET))
#define CHUNK_SIZE(size) (size+sizeof(r_malloc_chunk))

#define HDR_FLAG 0x464c4147

static UINT4 mem_usage;      /* Includes our header */
static UINT4 mem_stats[256]; /* Does not include our header */

void *r_malloc(int type, size_t size)
  {
    size_t total;
    r_malloc_chunk *chunk = 0;

    total=size+sizeof(r_malloc_chunk);

    if(!(chunk=(r_malloc_chunk*)malloc(total)))
      return(0);

#ifdef SANITY_CHECKS
    chunk->hdr=HDR_FLAG;
#endif
    chunk->type=type;
    chunk->size=size;

    mem_usage+=CHUNK_SIZE(size);
    mem_stats[type]+=size;

    return(chunk->memory);
  }

void *r_calloc(int type, size_t number, size_t size)
  {
    void *ret = 0;
    size_t total;

    total=number*size;

    if(!(ret=r_malloc(type,total)))
      return(0);

    memset(ret,0,size);

    return(ret);
  }

void r_free(void *ptr)
  {
    r_malloc_chunk *chunk = 0;

    if(!ptr) return;

    chunk=(r_malloc_chunk *)GET_CHUNK_ADDR_FROM_MEM_ADDR(ptr);
#ifdef SANITY_CHECKS
    assert(chunk->hdr==HDR_FLAG);
#endif

    mem_usage-=CHUNK_SIZE(chunk->size);
    mem_stats[chunk->type]-=chunk->size;

    free(chunk);
  }

#endif // ifndef NO_MALLOC_REPLACE

char *r_strdup(const char *str)
  {
    int len;
    char *nstr = 0;

    if(!str)
      return(0);

    len=strlen(str)+1;

    if(!(nstr=(char*)RMALLOC(len)))
      return(0);

    memcpy(nstr,str,len);

    return(nstr);
  }

