/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set sw=2 ts=8 et ft=cpp : */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "VideoEngine.h"

#include "libwebrtcglue/SystemTime.h"
#include "video_engine/desktop_capture_impl.h"

#ifdef MOZ_WIDGET_ANDROID
#  include "mozilla/jni/Utils.h"
#endif

#if defined(ANDROID)
namespace webrtc {
int32_t SetCaptureAndroidVM(JavaVM* javaVM);
}
#endif

namespace mozilla::camera {

#undef LOG
#undef LOG_ENABLED
mozilla::LazyLogModule gVideoEngineLog("VideoEngine");
#define LOG(args) MOZ_LOG(gVideoEngineLog, mozilla::LogLevel::Debug, args)
#define LOG_ENABLED() MOZ_LOG_TEST(gVideoEngineLog, mozilla::LogLevel::Debug)

#if defined(ANDROID)
int VideoEngine::SetAndroidObjects() {
  LOG(("%s", __PRETTY_FUNCTION__));

  JavaVM* const javaVM = mozilla::jni::GetVM();
  if (!javaVM || webrtc::SetCaptureAndroidVM(javaVM) != 0) {
    LOG(("Could not set capture Android VM"));
    return -1;
  }
#  ifdef WEBRTC_INCLUDE_INTERNAL_VIDEO_RENDER
  if (webrtc::SetRenderAndroidVM(javaVM) != 0) {
    LOG(("Could not set render Android VM"));
    return -1;
  }
#  endif
  return 0;
}
#endif

int32_t VideoEngine::CreateVideoCapture(const char* aDeviceUniqueIdUTF8,
                                        uint64_t aWindowID) {
  LOG(("%s", __PRETTY_FUNCTION__));
  MOZ_ASSERT(aDeviceUniqueIdUTF8);

  int32_t id = GenerateId();
  LOG(("CaptureDeviceType=%s id=%d", EnumValueToString(mCaptureDevType), id));

  for (auto& it : mSharedCapturers) {
    if (it.second.VideoCapture() &&
        it.second.VideoCapture()->CurrentDeviceName() &&
        strcmp(it.second.VideoCapture()->CurrentDeviceName(),
               aDeviceUniqueIdUTF8) == 0) {
      mIdToCapturerMap.emplace(id, CaptureHandle{.mCaptureEntryNum = it.first,
                                                 .mWindowID = aWindowID});
      return id;
    }
  }

  CaptureEntry entry = {-1, nullptr, nullptr};

  VideoCaptureFactory::CreateVideoCaptureResult capturer =
      mVideoCaptureFactory->CreateVideoCapture(id, aDeviceUniqueIdUTF8,
                                               mCaptureDevType);
  entry =
      CaptureEntry(id, std::move(capturer.mCapturer), capturer.mDesktopImpl);

  mSharedCapturers.emplace(id, std::move(entry));
  mIdToCapturerMap.emplace(
      id, CaptureHandle{.mCaptureEntryNum = id, .mWindowID = aWindowID});
  return id;
}

int VideoEngine::ReleaseVideoCapture(const int32_t aId) {
  bool found = false;

#ifdef DEBUG
  {
    auto it = mIdToCapturerMap.find(aId);
    MOZ_ASSERT(it != mIdToCapturerMap.end());
    (void)it;
  }
#endif

  for (auto& it : mIdToCapturerMap) {
    if (it.first != aId &&
        it.second.mCaptureEntryNum == mIdToCapturerMap[aId].mCaptureEntryNum) {
      // There are other tracks still using this hardware.
      found = true;
    }
  }

  if (!found) {
    WithEntry(aId, [&found](CaptureEntry& cap) {
      cap.mVideoCaptureModule = nullptr;
      found = true;
    });
    MOZ_ASSERT(found);
    if (found) {
      auto it = mSharedCapturers.find(mIdToCapturerMap[aId].mCaptureEntryNum);
      MOZ_ASSERT(it != mSharedCapturers.end());
      mSharedCapturers.erase(it);
    }
  }

  mIdToCapturerMap.erase(aId);
  return found ? 0 : (-1);
}

std::shared_ptr<webrtc::VideoCaptureModule::DeviceInfo>
VideoEngine::GetOrCreateVideoCaptureDeviceInfo() {
  LOG(("%s", __PRETTY_FUNCTION__));
  webrtc::Timestamp currentTime = webrtc::Timestamp::Micros(0);

  const char* capDevTypeName = EnumValueToString(mCaptureDevType);

  if (mDeviceInfo) {
    LOG(("Device cache available."));
    // Camera cache is invalidated by HW change detection elsewhere
    if (mCaptureDevType == CaptureDeviceType::Camera) {
      LOG(("returning cached CaptureDeviceInfo of type %s", capDevTypeName));
      return mDeviceInfo;
    }
    // Screen sharing cache is invalidated after the expiration time
    currentTime = WebrtcSystemTime();
    LOG(("Checking expiry, fetched current time of: %" PRId64,
         currentTime.ms()));
    LOG(("device cache expiration is %" PRId64, mExpiryTime.ms()));
    if (currentTime <= mExpiryTime) {
      LOG(("returning cached CaptureDeviceInfo of type %s", capDevTypeName));
      return mDeviceInfo;
    }
  }

  if (currentTime.IsZero()) {
    currentTime = WebrtcSystemTime();
    LOG(("Fetched current time of: %" PRId64, currentTime.ms()));
  }
  mExpiryTime = currentTime + webrtc::TimeDelta::Millis(kCacheExpiryPeriodMs);
  LOG(("new device cache expiration is %" PRId64, mExpiryTime.ms()));
  LOG(("creating a new VideoCaptureDeviceInfo of type %s", capDevTypeName));

#ifdef MOZ_WIDGET_ANDROID
  if (mCaptureDevType == CaptureDeviceType::Camera) {
    if (SetAndroidObjects()) {
      LOG(("VideoEngine::SetAndroidObjects Failed"));
      return mDeviceInfo;
    }
  }
#endif

  if (mDeviceInfo) {
    mDeviceInfo->DeRegisterVideoInputFeedBack(this);
  }

  mDeviceInfo = mVideoCaptureFactory->CreateDeviceInfo(mId, mCaptureDevType);

  if (mDeviceInfo && mCaptureDevType == CaptureDeviceType::Camera) {
    mDeviceInfo->RegisterVideoInputFeedBack(this);
  }

  LOG(("EXIT %s", __PRETTY_FUNCTION__));
  return mDeviceInfo;
}

void VideoEngine::ClearVideoCaptureDeviceInfo() {
  LOG(("%s", __PRETTY_FUNCTION__));
  if (mDeviceInfo) {
    mDeviceInfo->DeRegisterVideoInputFeedBack(this);
    OnDeviceChange();
  }
  mDeviceInfo.reset();
  mVideoCaptureFactory->Invalidate();
}

already_AddRefed<VideoEngine> VideoEngine::Create(
    const CaptureDeviceType& aCaptureDeviceType,
    RefPtr<VideoCaptureFactory> aVideoCaptureFactory) {
  LOG(("%s", __PRETTY_FUNCTION__));
  return do_AddRef(
      new VideoEngine(aCaptureDeviceType, std::move(aVideoCaptureFactory)));
}

VideoEngine::CaptureEntry::CaptureEntry(
    int32_t aCapnum, webrtc::scoped_refptr<webrtc::VideoCaptureModule> aCapture,
    webrtc::DesktopCaptureImpl* aDesktopImpl)
    : mCapnum(aCapnum),
      mVideoCaptureModule(std::move(aCapture)),
      mDesktopImpl(aDesktopImpl) {}

webrtc::scoped_refptr<webrtc::VideoCaptureModule>
VideoEngine::CaptureEntry::VideoCapture() {
  return mVideoCaptureModule;
}

mozilla::MediaEventSource<void>*
VideoEngine::CaptureEntry::CaptureEndedEvent() {
  if (!mDesktopImpl) {
    return nullptr;
  }
#if !defined(WEBRTC_ANDROID) && !defined(WEBRTC_IOS)
  return mDesktopImpl->CaptureEndedEvent();
#else
  return nullptr;
#endif
}

int32_t VideoEngine::CaptureEntry::Capnum() const { return mCapnum; }

bool VideoEngine::WithEntry(
    const int32_t entryCapnum,
    const std::function<void(CaptureEntry& entry)>&& fn) {
#ifdef DEBUG
  {
    auto it = mIdToCapturerMap.find(entryCapnum);
    MOZ_ASSERT(it != mIdToCapturerMap.end());
    (void)it;
  }
#endif

  auto it =
      mSharedCapturers.find(mIdToCapturerMap[entryCapnum].mCaptureEntryNum);
  MOZ_ASSERT(it != mSharedCapturers.end());
  if (it == mSharedCapturers.end()) {
    return false;
  }
  fn(it->second);
  return true;
}

bool VideoEngine::IsWindowCapturing(uint64_t aWindowID,
                                    const nsCString& aUniqueIdUTF8) {
  Maybe<int32_t> sharedId;
  for (auto& [id, entry] : mSharedCapturers) {
    if (entry.VideoCapture() && entry.VideoCapture()->CurrentDeviceName() &&
        strcmp(entry.VideoCapture()->CurrentDeviceName(),
               aUniqueIdUTF8.get()) == 0) {
      sharedId = Some(id);
      break;
    }
  }
  if (!sharedId) {
    return false;
  }
  for (auto& [id, handle] : mIdToCapturerMap) {
    if (handle.mCaptureEntryNum == *sharedId && handle.mWindowID == aWindowID) {
      return true;
    }
  }
  return false;
}

int32_t VideoEngine::GenerateId() {
  // XXX Something better than this (a map perhaps, or a simple boolean TArray,
  // given the number in-use is O(1) normally!)
  static int sId = 0;
  return mId = sId++;
}

void VideoEngine::OnDeviceChange() { mDeviceChangeEvent.Notify(); }

VideoEngine::VideoEngine(const CaptureDeviceType& aCaptureDeviceType,
                         RefPtr<VideoCaptureFactory> aVideoCaptureFactory)
    : mId(0),
      mCaptureDevType(aCaptureDeviceType),
      mVideoCaptureFactory(std::move(aVideoCaptureFactory)),
      mDeviceInfo(nullptr) {
  MOZ_ASSERT(mVideoCaptureFactory);
  LOG(("%s", __PRETTY_FUNCTION__));
  LOG(("Creating new VideoEngine with CaptureDeviceType %s",
       EnumValueToString(mCaptureDevType)));
}

VideoEngine::~VideoEngine() {
  MOZ_ASSERT(mSharedCapturers.empty());
  MOZ_ASSERT(mIdToCapturerMap.empty());
}

}  // namespace mozilla::camera
