/**
 * @file
 * @brief Zap definitions. See zap_info struct in beam.cc.
**/

#pragma once

/*
struct zap_info
{
    zap_type ztype;
    const char* name;
    dam_deducer* player_damage;
    tohit_deducer* player_tohit;       // Enchantments have power modifier here
    dam_deducer* monster_damage;
    tohit_deducer* monster_tohit;      // Enchantments have power modifier here
    colour_t colour;
    bool is_enchantment;
    beam_type flavour;
    dungeon_char_type glyph;
    bool can_beam;
    bool is_explosion;
    tileidx_t tile_beam;
    tileidx_t tile_explode;
}
*/

/// Boilerplate monster hex.
static zap_info _mon_hex_zap(zap_type ztype, beam_type beam,
                             colour_t colour = BLACK)
{
    return {
        ztype,
        "",
        nullptr,
        nullptr,
        nullptr,
        new tohit_calculator<0, 1, 1>,
        colour,
        true,
        beam,
        NUM_DCHAR_TYPES,
        false,
        false,
    };
}

static const zap_info zap_data[] =
{

{
    ZAP_NOXIOUS_CLOUD,
    "noxious blast",
    nullptr,
    nullptr,
    new dicedef_calculator<1, 0, 0, 1>,
    new tohit_calculator<18, 1, 25>,
    LIGHTGREEN,
    false,
    BEAM_POISON,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_POISONOUS_CLOUD,
    "blast of poison",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 25>,
    new tohit_calculator<18, 1, 25>,
    LIGHTGREEN,
    false,
    BEAM_POISON,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BOLT_OF_DEVASTATION,
    "bolt of devastation",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 20, 0, 1>,
    new tohit_calculator<15, 1, 30>,
    MAGENTA,
    false,
    BEAM_DEVASTATION,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_DOOM_BOLT,
    "doom bolt",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 9>,
    new tohit_calculator<15, 1, 30>,
    LIGHTRED,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_DOOM,
},

{
    ZAP_BOLT_OF_FLESH,
    "bolt of flesh",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 10>,
    new tohit_calculator<17, 1, 25>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_FLESH,
},

{
    ZAP_AWAKEN_FLESH,
    "surge of chaos",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 5, 1, 16>,
    new tohit_calculator<AUTOMATIC_HIT>,
    ETC_RANDOM,
    false,
    BEAM_CHAOS,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_CHAOS,
},

{
    ZAP_METAL_SPLINTERS,
    "spray of metal splinters",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 20, 1, 20>,
    new tohit_calculator<19, 1, 30>,
    CYAN,
    false,
    BEAM_FRAG,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_METAL_SPLINTERS,
},

{
    ZAP_CRYSTALLISING_SHOT,
    "crystallising shot",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 6, 1, 10>,
    new tohit_calculator<18, 1, 30>,
    LIGHTCYAN,
    false,
    BEAM_CRYSTALLISING,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_CRYSTAL_SPEAR,
},

{
    ZAP_SPLINTERSPRAY,
    "spray of wooden splinters",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 15, 1, 20>,
    new tohit_calculator<17, 1, 30>,
    BROWN,
    false,
    BEAM_FRAG,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_SPLINTERSPRAY,
},

{
    ZAP_THORN_VOLLEY,
    "volley of thorns",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 5, 1, 13>,
    new tohit_calculator<20, 1, 15>,
    BROWN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_SPLINTERSPRAY,
},

{
    ZAP_THROW_BARBS,
    "volley of spikes",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 13, 0, 1>,
    new tohit_calculator<27>,
    LIGHTGREY,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
},

{
    ZAP_HURL_SLUDGE,
    "toxic sludge",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 16>,
    new tohit_calculator<20, 1, 15>,
    LIGHTGREEN,
    false,
    BEAM_POISON_ARROW,
    DCHAR_FIRED_MISSILE,
    false,
    false,
},

{
    ZAP_CALL_DOWN_DAMNATION,
    "damnation",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 15, 0, 10>,
    new tohit_calculator<20>,
    LIGHTRED,
    false,
    BEAM_DAMNATION,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_DAMNATION,
},

{
    ZAP_MIASMA_BREATH,
    "foul vapour",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 5, 1, 24>,
    new tohit_calculator<17, 1, 20>,
    DARKGREY,
    false,
    BEAM_MIASMA,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_HOLY_BREATH,
    "blast of cleansing flame",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 0, 1, 12>,
    new tohit_calculator<18, 1, 25>,
    ETC_HOLY,
    false,
    BEAM_HOLY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_PETRIFYING_CLOUD,
    "blast of calcifying dust",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 6, 0, 1>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_PETRIFYING_CLOUD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_GHOSTLY_FIREBALL,
    "ghostly fireball",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 6, 1, 13>,
    new tohit_calculator<40>,
    CYAN,
    false,
    BEAM_NEG,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_GHOSTLY_FIREBALL,
},

{
    ZAP_FLASH_FREEZE,
    "icy flash freeze",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 10, 1, 17>,
    new tohit_calculator<16, 1, 13>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_CREEPING_FROST,
    "flash freeze",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 7, 1, 12>,
    new tohit_calculator<5, 1, 3>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_PERMAFROST_ERUPTION_COLD,
    "piercing cold",
    new calcdice_calculator<4, 6, 3, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<3, 7, 1, 20>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_PERMAFROST_ERUPTION_EARTH,
    "mass of rock",
    new calcdice_calculator<4, 6, 3, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<3, 7, 1, 20>,
    new tohit_calculator<AUTOMATIC_HIT>,
    LIGHTGREY,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
},

{
    ZAP_SPIT_LAVA,
    "glob of lava",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 10, 0, 1>,
    new tohit_calculator<20>,
    RED,
    false,
    BEAM_LAVA,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_MAGMA,
},

{
    ZAP_CHAOS_BREATH,
    "blast of chaos",
    nullptr,
    nullptr,
    new dicedef_calculator<1, 0, 1, 8>,
    new tohit_calculator<30>,
    ETC_RANDOM,
    false,
    BEAM_CHAOS,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_COLD_BREATH,
    "blast of cold",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 0, 1, 6>,
    new tohit_calculator<30>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_FIRE_BREATH,
    "blast of flame",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 0, 1, 6>,
    new tohit_calculator<30>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_SEARING_BREATH,
    "searing blast",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 0, 1, 9>,
    new tohit_calculator<30>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_ELECTRICAL_BOLT,
    "bolt of electricity",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 12>,
    new tohit_calculator<35>,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_FLAMING_CLOUD,
    "blast of flame",
    nullptr,
    nullptr,
    new dicedef_calculator<1, 0, 1, 8>,
    new tohit_calculator<30>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_THROW_FLAME,
    "puff of flame",
    new dicedef_calculator<2, 4, 1, 10>,
    new tohit_calculator<8, 1, 10>,
    new dicedef_calculator<3, 5, 1, 40>,
    new tohit_calculator<25, 1, 40>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_FLAME,
},

{
    ZAP_THROW_FROST,
    "puff of frost",
    new dicedef_calculator<2, 4, 1, 10>,
    new tohit_calculator<8, 1, 10>,
    new dicedef_calculator<3, 5, 1, 40>,
    new tohit_calculator<25, 1, 40>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_FROST,
},

{
    ZAP_BREATHE_ACID,
    "glob of acid",
    new dicedef_calculator<5, 4, 1, 3>,
    new tohit_calculator<7, 1, 6>,
    new dicedef_calculator<3, 14, 0, 1>,
    new tohit_calculator<20, 1, 4>,
    YELLOW,
    false,
    BEAM_ACID,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

_mon_hex_zap(ZAP_SLOW, BEAM_SLOW),

{
    ZAP_MEPHITIC,
    "stinking cloud",
    nullptr,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    GREEN,
    false,
    BEAM_MEPHITIC,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_MEPHITIC_FLASK,
    TILE_BOLT_DEFAULT_GREEN,
},

{
    ZAP_HASTE,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    BLACK,
    true,
    BEAM_HASTE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_MAGIC_DART,
    "magic dart",
    new dicedef_calculator<1, 3, 1, 5>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<3, 4, 1, 100>,
    new tohit_calculator<AUTOMATIC_HIT>,
    LIGHTMAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_MAGIC_DART,
},

_mon_hex_zap(ZAP_PARALYSE, BEAM_PARALYSIS),
_mon_hex_zap(ZAP_VEX, BEAM_VEX),

{
    ZAP_BOLT_OF_FIRE,
    "bolt of fire",
    new calcdice_calculator<6, 18, 2, 3>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 8, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BOLT_OF_COLD,
    "bolt of cold",
    new calcdice_calculator<6, 18, 2, 3>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 8, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{ // Used only by phial of floods
    ZAP_PRIMAL_WAVE,
    "torrent of water",
    new calcdice_calculator<4, 14, 3, 5>,
    new tohit_calculator<10, 1, 25>,
    // Water attack is weaker than the pure elemental damage attacks, but also
    // less resistible.
    new dicedef_calculator<3, 6, 1, 10>,
    // Huge wave of water is hard to dodge.
    new tohit_calculator<14, 1, 35>,
    LIGHTBLUE,
    false,
    BEAM_WATER,
    DCHAR_WAVY,
    false,
    false,
},

_mon_hex_zap(ZAP_CONFUSE, BEAM_CONFUSION),
_mon_hex_zap(ZAP_TUKIMAS_DANCE, BEAM_TUKIMAS_DANCE),

{
    ZAP_INVISIBILITY,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    BLACK,
    true,
    BEAM_INVISIBILITY,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_DIG,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    BLACK,
    true,
    BEAM_DIGGING,
    NUM_DCHAR_TYPES,
    true,
    false,
},

{
    ZAP_HELLFIRE_MORTAR_DIG,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    RED,
    true,
    BEAM_DIGGING,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_FASTROOT,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    BROWN,
    true,
    BEAM_ROOTS,
    DCHAR_FIRED_BURST,
    false,
    true,
},

{
    ZAP_FIREBALL,
    "fireball",
    new calcdice_calculator<3, 10, 1, 2>,
    new tohit_calculator<40>,
    new dicedef_calculator<3, 7, 1, 10>,
    new tohit_calculator<40>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_FLAME,
    TILE_BOLT_DEFAULT_RED,
},

{
    ZAP_BLASTMOTE,
    "concussive blast",
    new calcdice_calculator<2, 20, 1, 3>,
    new tohit_calculator<40>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    false,
    true,
},


{
    ZAP_FLAME_WAVE,
    "flame wave",
    new calcdice_calculator<2, 9, 1, 3>,
    new tohit_calculator<40>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_IGNITION,
    "fireball",
    new calcdice_calculator<3, 10, 1, 3>, // less than fireball
    new tohit_calculator<40>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_FIRE,
    NUM_DCHAR_TYPES,
    false,
    true,
},

{
    ZAP_TELEPORT_OTHER,
    "",
    nullptr,
    new tohit_calculator<0, 3, 2>,
    nullptr,
    new tohit_calculator<0, 1, 1>,
    BLACK,
    true,
    BEAM_TELEPORT,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_LIGHTNING_BOLT,
    "bolt of lightning",
    new calcdice_calculator<1, 11, 3, 5>,
    new tohit_calculator<7, 1, 40>,
    new dicedef_calculator<3, 10, 1, 17>,
    new tohit_calculator<16, 1, 40>,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

_mon_hex_zap(ZAP_POLYMORPH, BEAM_POLYMORPH),

{
    ZAP_VENOM_BOLT,
    "bolt of poison",
    new calcdice_calculator<4, 16, 3, 5>,
    new tohit_calculator<8, 1, 20>,
    new dicedef_calculator<3, 6, 1, 13>,
    new tohit_calculator<19, 1, 20>,
    LIGHTGREEN,
    false,
    BEAM_POISON,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BOLT_OF_DRAINING,
    "bolt of negative energy",
    new calcdice_calculator<4, 15, 3, 5>,
    new tohit_calculator<8, 1, 20>,
    new dicedef_calculator<3, 9, 1, 13>,
    new tohit_calculator<17, 1, 25>,
    DARKGREY,
    false,
    BEAM_NEG,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_DRAIN,
},

{
    ZAP_KISS_OF_DEATH,
    "fragment of death",
    new calcdice_calculator<2, 11, 3, 5>, // caps at 2d13... spooky
    new tohit_calculator<AUTOMATIC_HIT>, // XXX: should we let this miss?
    new calcdice_calculator<2, 13, 2, 7>,
    new tohit_calculator<AUTOMATIC_HIT>,
    DARKGREY,
    false,
    BEAM_NEG,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_LEHUDIBS_CRYSTAL_SPEAR,      // was splinters
    "crystal spear",
    new calcdice_calculator<10, 23, 1, 1>,
    new tohit_calculator<10, 1, 15>,
    new dicedef_calculator<3, 16, 1, 10>,
    new tohit_calculator<22, 1, 20>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_CRYSTAL_SPEAR,
},

_mon_hex_zap(ZAP_CHARMING, BEAM_CHARM),

{
    ZAP_PAIN,
    "",
    nullptr,
    nullptr,
    new dicedef_calculator<1, 7, 1, 20>,
    new tohit_calculator<0, 1, 1>,
    BLACK,
    true,
    BEAM_PAIN,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_STICKY_FLAME,
    "sticky flame",
    new dicedef_calculator<2, 4, 1, 9>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<2, 2, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_PYRE_ARROW,
    "pyre arrow",
    nullptr,
    nullptr,
    // 0 dice so 'does no damage' is suppresssed
    new dicedef_calculator<0, 0, 0, 1>,
    new tohit_calculator<18, 1, 12>,
    RED,
    false,
    BEAM_STICKY_FLAME,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_DISPEL_UNDEAD,
    "",
    new calcdice_calculator<3, 20, 3, 4>,
    new tohit_calculator<0, 3, 2>,
    new dicedef_calculator<3, 3, 1, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    BLACK,
    true,
    BEAM_DISPEL_UNDEAD,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_DISPEL_UNDEAD_RANGE,
    "",
    new calcdice_calculator<3, 20, 3, 4>,
    new tohit_calculator<0, 3, 2>,
    new dicedef_calculator<3, 6, 1, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    BLACK,
    true,
    BEAM_DISPEL_UNDEAD,
    NUM_DCHAR_TYPES,
    false,
    false,
},

_mon_hex_zap(ZAP_BANISHMENT, BEAM_BANISH),

{
    ZAP_STING,
    "sting",
    new dicedef_calculator<1, 3, 1, 4>,
    new tohit_calculator<8, 1, 5>,
    new dicedef_calculator<2, 4, 1, 10>,
    new tohit_calculator<12, 1, 5>,
    GREEN,
    false,
    BEAM_POISON_ARROW,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_STING,
},

{
    ZAP_HURL_DAMNATION,
    "damnation",
    new calcdice_calculator<3, 8, 3, 5>,
    new tohit_calculator<20, 1, 10>,
    new dicedef_calculator<3, 20, 0, 10>,
    new tohit_calculator<24>,
    LIGHTRED,
    false,
    BEAM_DAMNATION,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_DAMNATION,
},

{
    ZAP_IRON_SHOT,
    "iron shot",
    new calcdice_calculator<9, 15, 3, 4>,
    new tohit_calculator<7, 1, 15>,
    new dicedef_calculator<3, 8, 1, 9>,
    new tohit_calculator<20, 1, 25>,
    LIGHTCYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_IRON_SHOT,
},

{
    ZAP_BOMBARD,
    "iron shot",
    new calcdice_calculator<9, 13, 2, 3>,
    new tohit_calculator<7, 1, 15>,
    new dicedef_calculator<3, 8, 1, 9>,
    new tohit_calculator<20, 1, 25>,
    LIGHTCYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_IRON_SHOT,
},

{
    ZAP_STONE_ARROW,
    "stone arrow",
    new dicedef_calculator<3, 7, 1, 8>,
    new tohit_calculator<8, 1, 10>,
    new dicedef_calculator<3, 5, 1, 10>,
    new tohit_calculator<14, 1, 35>,
    LIGHTGREY,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_STONE_ARROW,
},

{
    ZAP_SHOCK,
    "zap",
    new dicedef_calculator<1, 3, 1, 4>,
    new tohit_calculator<8, 1, 7>,
    new dicedef_calculator<1, 8, 1, 6>,
    new tohit_calculator<17, 1, 20>,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,             // beams & reflects
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_ZAP,
},

{
    ZAP_STUNNING_BURST,
    "stunning burst",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 8, 1, 20>,
    new tohit_calculator<15, 1, 10>,
    LIGHTCYAN,
    false,
    BEAM_STUN_BOLT,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_ORB_OF_ELECTRICITY,
    "orb of electricity",
    new calcdice_calculator<0, 13, 4, 5>,
    new tohit_calculator<40>,
    new dicedef_calculator<3, 7, 1, 9>,
    new tohit_calculator<40>,
    LIGHTBLUE,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_SPIT_POISON,
    "splash of poison",
    new dicedef_calculator<1, 4, 1, 2>,
    new tohit_calculator<5, 1, 6>,
    new dicedef_calculator<2, 4, 1, 10>,
    new tohit_calculator<16, 1, 20>,
    GREEN,
    false,
    BEAM_POISON,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_DEBUGGING_RAY,
    "debugging ray",
    new dicedef_calculator<AUTOMATIC_HIT, 1, 0, 1>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_DEBUG,
    false,
    false,
},

{
    ZAP_BREATHE_FIRE,
    "fiery breath",
    new dicedef_calculator<3, 4, 1, 3>,
    new tohit_calculator<8, 1, 6>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_GLACIAL_BREATH,
    "glacial breath",
    new dicedef_calculator<3, 6, 4, 3>,
    new tohit_calculator<12, 1, 1>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_COMBUSTION_BREATH,
    "explosive embers",
    new dicedef_calculator<0, 0, 0, 1>, // deals damage through explosions
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_MISSILE, // To avoid printing needless messages for non-damaging hits
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BREATHE_POISON,
    "poison gas",
    new dicedef_calculator<3, 2, 1, 6>,
    new tohit_calculator<6, 1, 6>,
    nullptr,
    nullptr,
    GREEN,
    false,
    BEAM_POISON,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_SPIT_ACID,
    "splash of acid",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 14, 0, 1>,
    new tohit_calculator<20, 1, 4>,
    YELLOW,
    false,
    BEAM_ACID,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_AGONY,
    "agony",
    nullptr,
    new tohit_calculator<0, 5, 1>,
    nullptr,
    new tohit_calculator<0, 1, 1>,
    BLACK,
    true,
    BEAM_AGONY,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_CURSE_OF_AGONY,
    "curse of agony",
    nullptr,
    new tohit_calculator<0, 5, 2>,
    nullptr,
    new tohit_calculator<0, 1, 1>,
    BLACK,
    true,
    BEAM_CURSE_OF_AGONY,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_VAMPIRIC_DRAINING,
    "",
    new dicedef_calculator<2, 6, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<2, 6, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    BLACK,
    true,
    BEAM_VAMPIRIC_DRAINING,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_MINDBURST,
    "",
    new calcdice_calculator<3, 15, 3, 4>,
    new tohit_calculator<0, 5, 2>,
    new calcdice_calculator<1, 12, 1, 4>,
    new tohit_calculator<50, 0, 1>,
    BLACK,
    true,
    BEAM_MINDBURST,
    NUM_DCHAR_TYPES,
    true,
    false,
},

{
    ZAP_BREATHE_STEAM,
    "ball of steam",
    new dicedef_calculator<3, 4, 2, 3>,
    new tohit_calculator<10, 1, 10>,
    new dicedef_calculator<3, 7, 1, 15>,
    new tohit_calculator<20, 1, 20>,
    LIGHTGREY,
    false,
    BEAM_STEAM,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_THROW_ICICLE,
    "shard of ice",
    new calcdice_calculator<3, 10, 1, 2>,
    new tohit_calculator<9, 1, 12>,
    new dicedef_calculator<3, 8, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_ICICLE,
},

{
    ZAP_CORONA,
    "",
    nullptr,
    nullptr,
    nullptr,
    new tohit_calculator<0, 1, 1>,
    BLUE,
    true,
    BEAM_CORONA,
    NUM_DCHAR_TYPES,
    false,
    false,
},

_mon_hex_zap(ZAP_HIBERNATION, BEAM_HIBERNATION),

{
    ZAP_SANDBLAST,
    "blast of sand",
    new dicedef_calculator<2, 4, 1, 3>,
    new tohit_calculator<15, 1, 10>,
    new dicedef_calculator<3, 7, 1, 18>,
    new tohit_calculator<20, 1, 40>,
    BROWN,
    false,
    BEAM_FRAG,
    DCHAR_FIRED_BOLT,
    false,
    false,
    TILE_BOLT_SANDBLAST,
},

{
    ZAP_BOLT_OF_MAGMA,
    "bolt of magma",
    new calcdice_calculator<4, 16, 2, 3>,
    new tohit_calculator<8, 1, 25>,
    new dicedef_calculator<3, 8, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    RED,
    false,
    BEAM_LAVA,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_MAGMA,
},

{
    ZAP_MOMENTUM_STRIKE,
    "concentrated momentum",
    new calcdice_calculator<3, 24, 1, 6>,
    new tohit_calculator<9, 1, 12>,
    nullptr,
    nullptr,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_POISON_ARROW,
    "poison arrow",
    new calcdice_calculator<4, 15, 1, 1>,
    new tohit_calculator<5, 1, 10>,
    new dicedef_calculator<3, 7, 1, 12>,
    new tohit_calculator<20, 1, 25>,
    LIGHTGREEN,
    false,
    BEAM_POISON_ARROW,             // extra damage
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_POISON_ARROW,
},

_mon_hex_zap(ZAP_PETRIFY, BEAM_PETRIFY),
_mon_hex_zap(ZAP_PORKALATOR, BEAM_PORKALATOR, RED),
_mon_hex_zap(ZAP_SLEEP, BEAM_SLEEP),

{
    ZAP_IOOD,
    "",
    nullptr,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_DESTRUCTION,
    NUM_DCHAR_TYPES, // no dchar, to get bolt.glyph == 0,
                     // hence invisible
    true,
    false,
},

{
    ZAP_NOXIOUS_BREATH,
    "blast of choking fumes",
    new dicedef_calculator<0, 1, 1, 1>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    GREEN,
    false,
    BEAM_MEPHITIC,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_INNER_FLAME,
    "",
    nullptr,
    new tohit_calculator<0, 3, 1>,
    nullptr,
    nullptr,
    BLACK,
    true,
    BEAM_INNER_FLAME,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_FORCE_LANCE,
    "lance of force",
    new dicedef_calculator<3, 6, 1, 6>,
    new tohit_calculator<20, 1, 7>,
    new dicedef_calculator<3, 6, 1, 15>,
    new tohit_calculator<20, 1, 20>,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_FORCE_LANCE,
},

{
    ZAP_SEARING_RAY,
    "searing ray",
    new calcdice_calculator<2, 9, 1, 7>,
    new tohit_calculator<11, 1, 8>,
    new calcdice_calculator<2, 7, 1, 4>,
    new tohit_calculator<11, 1, 7>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_SEARING_RAY,
},

{
    ZAP_REBOUNDING_BLAZE,
    "rebounding bolt of fire",
    new calcdice_calculator<6, 18, 2, 3>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 11, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_REBOUNDING_CHILL,
    "rebounding bolt of cold",
    new calcdice_calculator<6, 18, 2, 3>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 11, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},


{
    ZAP_QUICKSILVER_BOLT,
    "bolt of dispelling energy",
    new calcdice_calculator<6, 15, 2, 3>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 7, 1, 14>,
    new tohit_calculator<16, 1, 25>,
    ETC_RANDOM,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_CORROSIVE_BOLT,
    "bolt of acid",
    new calcdice_calculator<4, 13, 3, 5>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 7, 1, 10>,
    new tohit_calculator<17, 1, 25>,
    YELLOW,
    false,
    BEAM_ACID,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BOLT_OF_LIGHT,
    "bolt of light",
    new calcdice_calculator<4, 13, 3, 5>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 7, 1, 12>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_LIGHT,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_LIGHT,
},

{
    ZAP_UNRAVELLING,
    "unravelling",
    new dicedef_calculator<3, 3, 1, 6>,
    new tohit_calculator<0, 1, 1>,
    nullptr,
    nullptr,
    ETC_MUTAGENIC,
    true,
    BEAM_UNRAVELLING,
    NUM_DCHAR_TYPES,
    false,
    false,
    TILE_BOLT_IRRADIATE,
},

{
    ZAP_ICEBLAST,
    "iceblast",
    new calcdice_calculator<3, 14, 3, 5>,
    new tohit_calculator<40>,
    new dicedef_calculator<3, 6, 1, 12>, // slightly weaker than magma/fireball
    new tohit_calculator<40>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_MISSILE,
    false,
    true,
    TILE_BOLT_ICICLE,
    TILE_BOLT_ICEBLAST,
},

{
    ZAP_SLUG_DART,
    "slug dart",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 3, 1, 25>,
    new tohit_calculator<14, 1, 35>,
    CYAN, // match slug's own colour
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_STONE_ARROW,
},

{
    ZAP_CHAIN_LIGHTNING,
    "bolt of lightning",
    new calcdice_calculator<3, 0, 2, 1>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<3, 27, 1, 12>,
    new tohit_calculator<AUTOMATIC_HIT>,
    LIGHTCYAN,
    false,
    BEAM_THUNDER,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BLINKBOLT,
    "living lightning",
    new dicedef_calculator<1, 11, 3, 5>,
    new tohit_calculator<16, 1, 4>,
    new dicedef_calculator<2, 10, 1, 17>,
    new tohit_calculator<16, 1, 40>,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_FREEZING_BLAST,
    "freezing blast",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 9, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_COLD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BECKONING_GALE,
    "beckoning whirlwind",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 4, 1, 20>,
    nullptr,
    BLACK,
    false,
    BEAM_MMISSILE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_HARPOON_SHOT,
    "harpoon shot",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 7, 1, 20>,
    new tohit_calculator<17, 1, 25>,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_HARPOON_SHOT,
},

{
    ZAP_BOLAS,
    "whirling bolas",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 5, 1, 20>,
    new tohit_calculator<20, 1, 20>,
    BROWN,
    false,
    BEAM_BOLAS,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_BOLAS,
},

_mon_hex_zap(ZAP_DIMENSION_ANCHOR, BEAM_DIMENSION_ANCHOR),
_mon_hex_zap(ZAP_VULNERABILITY, BEAM_VULNERABILITY),
_mon_hex_zap(ZAP_SENTINEL_MARK, BEAM_SENTINEL_MARK),
_mon_hex_zap(ZAP_VIRULENCE, BEAM_VIRULENCE),
_mon_hex_zap(ZAP_SAP_MAGIC, BEAM_SAP_MAGIC),
_mon_hex_zap(ZAP_DIMINISH_SPELLS, BEAM_DIMINISH_SPELLS),
_mon_hex_zap(ZAP_DRAIN_MAGIC, BEAM_DRAIN_MAGIC),
_mon_hex_zap(ZAP_VITRIFY, BEAM_VITRIFY),

{
    ZAP_BECKONING,
    "beckoning",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    ETC_WARP,
    true,
    BEAM_BECKONING,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_FIRE_STORM,
    "great blast of fire",
    new calcdice_calculator<8, 5, 1, 1>,
    new tohit_calculator<40>,
    new calcdice_calculator<8, 5, 1, 2>,
    new tohit_calculator<40>,
    RED,
    false,
    BEAM_LAVA, // partially unaffected by rF
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_FIRE_STORM,
},

{
    ZAP_MALMUTATE,
    "malmutate",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    ETC_MUTAGENIC,
    true,
    BEAM_MALMUTATE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_VILE_CLUTCH,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    GREEN,
    true,
    BEAM_VILE_CLUTCH,
    DCHAR_FIRED_BURST,
    true,
    false,
},

{
    ZAP_REFRIGERATE,
    "refrigeration",
    new calcdice_calculator<4, 30, 4, 9>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new calcdice_calculator<4, 34, 9, 25>,
    new tohit_calculator<AUTOMATIC_HIT>,
    LIGHTCYAN,
    false,
    BEAM_COLD,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_DRAIN_LIFE,
    "drain life",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 6, 7, 10>,
    new tohit_calculator<40>,
    DARKGREY,
    false,
    BEAM_NEG,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_SONIC_WAVE,
    "sonic wave",
    new calcdice_calculator<2, 5, 1, 5>,
    new tohit_calculator<40>,
    new calcdice_calculator<2, 5, 1, 5>,
    new tohit_calculator<40>,
    YELLOW,
    false,
    BEAM_MMISSILE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_THROW_PIE,
    "klown pie",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 20, 0, 1>,
    new tohit_calculator<16, 1, 25>,
    BROWN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_PIE,
},

{
    ZAP_FOXFIRE,
    "foxfire",
    new dicedef_calculator<1, 4, 1, 5>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<3, 3, 1, 12>,
    new tohit_calculator<AUTOMATIC_HIT>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_BOLT,
    false,
    false,
},

{
    ZAP_TREMORSTONE,
    "burst of rock shards",
    new dicedef_calculator<6, 6, 0, 1>,
    new tohit_calculator<40>,
    nullptr,
    nullptr,
    BROWN,
    false,
    BEAM_FRAG,
    DCHAR_FIRED_BURST,
    false,
    true,
},

{
    ZAP_HAILSTORM,
    "hail",
    new calcdice_calculator<3, 15, 1, 3>,
    new tohit_calculator<9, 1, 10>,
    nullptr,
    nullptr,
    ETC_ICE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_BURST,
    false,
    false,
},

{
    ZAP_MYSTIC_BLAST,
    "mystic blast",
    new calcdice_calculator<2, 6, 1, 3>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    LIGHTMAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_MYSTIC_BLAST,
},

{
    ZAP_SWORD_BEAM,
    "bolt of energy",
    new calcdice_calculator<1, 40, 0, 1>,
    new tohit_calculator<5>,
    new calcdice_calculator<1, 40, 0, 1>,
    new tohit_calculator<5>,
    RED,
    false,
    BEAM_DESTRUCTION,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_MIGHT,
    "",
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    BLACK,
    true,
    BEAM_MIGHT,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_ENFEEBLE,
    "",
    nullptr,
    new tohit_calculator<0, 4, 1>,
    nullptr,
    new tohit_calculator<0, 2, 1>,
    BLACK,
    true,
    BEAM_ENFEEBLE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_SOUL_SPLINTER,
    "",
    nullptr,
    new tohit_calculator<6, 1, 2>,
    nullptr,
    new tohit_calculator<0, 4, 1>,
    BLACK,
    true,
    BEAM_SOUL_SPLINTER,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_PLASMA,
    "fiery plasma",
    new dicedef_calculator<1, 10, 11, 20>,
    new tohit_calculator<7, 1, 40>,
    new dicedef_calculator<3, 10, 1, 17>,
    new tohit_calculator<16, 1, 40>,
    RED,
    false,
    BEAM_FIRE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_PLASMA_LIGHTNING, // please keep damage dice identical to ZAP_PLASMA
    "bolt of lightning",
    new dicedef_calculator<1, 10, 11, 20>,
    new tohit_calculator<7, 1, 40>,
    new dicedef_calculator<3, 10, 1, 17>,
    new tohit_calculator<16, 1, 40>,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_MALIGN_OFFERING,
    "malign ofering",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 4, 1, 10>,
    nullptr,
    DARKGRAY,
    true,
    BEAM_MALIGN_OFFERING,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_HURL_TORCHLIGHT,
    "umbral torchlight",
    new dicedef_calculator<3, 7, 1, 5>,
    new tohit_calculator<40>,
    new dicedef_calculator<3, 6, 1, 10>,
    new tohit_calculator<40>,
    CYAN,
    false,
    BEAM_UMBRAL_TORCHLIGHT,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_UMBRAL_TORCHLIGHT,
},

{
    ZAP_WARP_SPACE,
    "rupture",
    new calcdice_calculator<3, 12, 3, 5>,
    new tohit_calculator<40>,
    new dicedef_calculator<3, 4, 1, 12>,
    new tohit_calculator<40>,
    LIGHTMAGENTA,
    false,
    BEAM_WARPING,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_WARP_SPACE,
},

{
    ZAP_SOJOURNING,
    "destabilizing rupture",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 7, 1, 12>,
    new tohit_calculator<40>,
    LIGHTMAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_WARP_SPACE,
},

{
    ZAP_CAUSTIC_BREATH,
    "acid spray",
    new dicedef_calculator<3, 3, 1, 1>,
    new tohit_calculator<30>,
    new dicedef_calculator<3, 3, 1, 1>,
    new tohit_calculator<30>,
    YELLOW,
    false,
    BEAM_ACID,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_NULLIFYING_BREATH,
    "nullifying energy",
    new dicedef_calculator<3, 4, 1, 3>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_MUD_BREATH,
    "ball of mud",
    new dicedef_calculator<3, 3, 1, 1>,
    new tohit_calculator<30>,
    nullptr,
    nullptr,
    BROWN,
    false,
    BEAM_MISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_GALVANIC_BREATH,
    "arc of electricity",
    new dicedef_calculator<3, 3, 9, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_AUTODAZZLE,
    "blinding ray",
    new dicedef_calculator<0, 1, 0, 1>,
    new tohit_calculator<10, 1, 25>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_LIGHT,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_LIGHT,
},

{
    ZAP_GRAVITAS,
    "crushing gravity",
    new dicedef_calculator<3, 6, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    LIGHTMAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_MAGNAVOLT,
    "arc of electricity",
    new dicedef_calculator<4, 9, 1, 10>,
    new tohit_calculator<11, 1, 20>,
    nullptr,
    nullptr,
    LIGHTCYAN,
    false,
    BEAM_ELECTRICITY,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_FULSOME_FUSILLADE,
    "volatile concoction",
    new dicedef_calculator<3, 5, 1, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_RIMEBLIGHT,
    "",
    nullptr,
    new tohit_calculator<0, 1, 3>,
    nullptr,
    new tohit_calculator<0, 1, 3>,
    BLUE,
    true,
    BEAM_RIMEBLIGHT,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_RIMEBLIGHT_SHARDS,
    "shards of blighted ice",
    new dicedef_calculator<3, 6, 1, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    ETC_ICE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_BURST,
    false,
    true,
},

{
    ZAP_HOARFROST_BULLET,
    "shard of frigid ice",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 16>,
    new tohit_calculator<13, 1, 35>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_ICICLE,
},

{
    ZAP_HOARFROST_BULLET_FINALE,
    "salvo of frigid ice",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 6, 1, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_MISSILE,
    false,
    true,
    TILE_BOLT_ICICLE,
    TILE_BOLT_ICEBLAST,
},

{
    ZAP_FLASHING_BALESTRA,
    "flashing steel",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 7, 1, 12>,
    new tohit_calculator<20, 1, 25>,
    LIGHTCYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
},

{
    ZAP_PHANTOM_BLITZ,
    "phantom echo",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 9, 1, 11>,
    new tohit_calculator<20, 1, 30>,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_PHANTOM_BLITZ,
},

{
    ZAP_SHADOW_SHARD,
    "shard of hardened shadow",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 7, 1, 9>,
    new tohit_calculator<20, 1, 10>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_SHADOW_SHARD,
},

{
    ZAP_SHADOW_BOLT,
    "ray of shadow",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 6, 1, 13>,
    new tohit_calculator<20, 1, 15>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_SHADOW_BEAM,
},

{
    ZAP_SHADOWBALL,
    "shadowball",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 4, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_SHADOW_BLAST,
},

{
    ZAP_CREEPING_SHADOW,
    "creeping shadow",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 4, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_SHADOW_BLAST,
},

{
    ZAP_SHADOW_JOLT,
    "shadowy lightning",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 4, 1, 16>,
    new tohit_calculator<15, 1, 20>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_SHADOW_DRAIN,
    "wave of devouring shadow",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 3, 1, 18>,
    new tohit_calculator<AUTOMATIC_HIT>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_SHADOW_TORPOR,
    "",
    nullptr,
    new tohit_calculator<0, 1, 3>,
    nullptr,
    new tohit_calculator<0, 1, 3>,
    BLUE,
    true,
    BEAM_SHADOW_TORPOR,
    NUM_DCHAR_TYPES,
    true,
    false,
},

{
    ZAP_SHADOW_SHOT,
    "shadow bullet",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 3, 1, 17>,
    new tohit_calculator<15, 1, 15>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_SHADOW_SHOT,
},

{
    ZAP_GRAVE_CLAW,
    "jagged bones",
    new dicedef_calculator<2, 7, 1, 7>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<2, 4, 1, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_MMISSILE,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_UNLEASH_DESTRUCTION,
    "gout of destruction",
    new dicedef_calculator<3, 5, 1, 2>,
    new tohit_calculator<10, 1, 7>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_UNLEASH_DESTRUCTION_PIERCING,
    "gout of destruction",
    new dicedef_calculator<3, 5, 1, 2>,
    new tohit_calculator<10, 1, 7>,
    nullptr,
    nullptr,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_BLOOD_ARROW,
    "blood arrow",
    new dicedef_calculator<4, 3, 27, 100>,
    new tohit_calculator<20, 1, 7>,
    nullptr,
    nullptr,
    LIGHTRED,
    false,
    BEAM_BLOODRITE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_BLOOD_ARROW,
},

{
    ZAP_SPIKE_LAUNCHER,
    "rocky spike",
    new dicedef_calculator<2, 3, 1, 12>,
    new tohit_calculator<12, 1, 10>,
    nullptr,
    nullptr,
    LIGHTGREY,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
},

{
    ZAP_KINETIC_GRAPNEL,
    "kinetic grapnel",
    new dicedef_calculator<1, 4, 4, 25>,
    new tohit_calculator<8, 1, 5>,
    nullptr,
    nullptr,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_SHRED,
    "shred",
    nullptr,
    nullptr,
    new dicedef_calculator<4, 4, 1, 22>,
    new tohit_calculator<AUTOMATIC_HIT>,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_PARAGON_IMPACT,
    "shockwave",
    new dicedef_calculator<4, 10, 1, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<4, 10, 1, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_MONARCH_DETONATION,
    "detonation",
    new dicedef_calculator<4, 5, 1, 8>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new calcdice_calculator<4, 7, 2, 5>,
    new tohit_calculator<AUTOMATIC_HIT>,
    RED,
    false,
    BEAM_LAVA,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_DEPLOY_BOMBLET,
    "bomblet",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 3, 1, 30>,
    new tohit_calculator<14, 1, 35>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_BOMBLET_LAUNCH,
},

{
    ZAP_SPLINTERFROST_FRAGMENT,
    "salvo of icicles",
    new calcdice_calculator<6, 24, 4, 5>,
    new tohit_calculator<10, 1, 25>,
    new dicedef_calculator<3, 8, 1, 11>,
    new tohit_calculator<17, 1, 25>,
    WHITE,
    false,
    BEAM_ICE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_ICICLE_SALVO,
},

{
    ZAP_PERCUSSIVE_TEMPERING,
    "blast",
    new dicedef_calculator<4, 4, 1, 12>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<4, 4, 1, 16>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_FORTRESS_BLAST,
    "fortress blast",
    new dicedef_calculator<6, 4, 3, 17>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<6, 4, 1, 6>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_FORTRESS_BLAST,
},

{
    ZAP_SEISMIC_STOMP,
    "seismic shockwave",
    nullptr,
    nullptr,
    new dicedef_calculator<4, 3, 1, 23>,
    new tohit_calculator<AUTOMATIC_HIT>,
    YELLOW,
    false,
    BEAM_SEISMIC,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_RENDING_SLASH,
    "condensed magic",
    nullptr,
    nullptr,
    new dicedef_calculator<2, 6, 1, 12>,
    new tohit_calculator<30>,
    MAGENTA,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_MAGMA_BARRAGE,
    "bolt of magma",
    nullptr,
    nullptr,
    new dicedef_calculator<4, 4, 2, 21>,
    new tohit_calculator<17, 1, 25>,
    RED,
    false,
    BEAM_LAVA,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_MAGMA,
},

{
    ZAP_MERCURY_ARROW,
    "glob of mercury",
    new calcdice_calculator<2, 11, 1, 4>,
    new tohit_calculator<12, 1, 5>,
    new dicedef_calculator<2, 4, 1, 10>,
    new tohit_calculator<12, 1, 5>,
    GREEN,
    false,
    BEAM_MERCURY,
    DCHAR_FIRED_ZAP,
    false,
    false,
    TILE_BOLT_STING,
},

{
    ZAP_FREEZE,
    "freeze",
    new dicedef_calculator<1, 3, 3, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<1, 3, 3, 10>,
    new tohit_calculator<AUTOMATIC_HIT>,
    WHITE,
    false,
    BEAM_COLD,
    NUM_DCHAR_TYPES,
    false,
    false,
},

{
    ZAP_RAVENOUS_SWARM,
    "swarm of vampire bats",
    nullptr,
    nullptr,
    new dicedef_calculator<0, 1, 1, 1>,
    new tohit_calculator<AUTOMATIC_HIT>,
    BLUE,
    false,
    BEAM_BAT_CLOUD,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_WATERY_GRAVE,
    "grasping water",
    new calcdice_calculator<0, 1, 1, 1>,    // Overriden by Form::get_special_damage()
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    LIGHTBLUE,
    false,
    BEAM_WATER,
    DCHAR_WAVY,
    false,
    false,
},

{
    ZAP_RUST_BREATH,
    "billowing rust",
    nullptr,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    YELLOW,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
},

{
    ZAP_GHOSTLY_SACRIFICE,
    "ghostly fireball",
    nullptr,
    nullptr,
    new dicedef_calculator<5, 6, 1, 13>,
    new tohit_calculator<40>,
    DARKGREY,
    false,
    BEAM_NEG,
    DCHAR_FIRED_ZAP,
    false,
    true,
    TILE_BOLT_GHOSTLY_FIREBALL,
},

// Name and flavor filled in via cast_golden_breath
{
    ZAP_GOLDEN_BREATH,
    "buggy blast",
    new dicedef_calculator<3, 2, 4, 3>,
    new tohit_calculator<20, 1, 5>,
    nullptr,
    nullptr,
    RED,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_ZAP,
    true,
    false,
    TILE_BOLT_FROSTFIRE
},

_mon_hex_zap(ZAP_ILL_OMEN, BEAM_ILL_OMEN),

{
    ZAP_WARP_BODY,
    "",
    nullptr,
    nullptr,
    new dicedef_calculator<1, 3, 1, 18>,
    new tohit_calculator<40>,
    DARKGREY,
    true,
    BEAM_WARP_BODY,
    DCHAR_FIRED_ZAP,
    false,
    false,
},

{
    ZAP_CONTAM_EXPLOSION,
    "magical storm",
    new dicedef_calculator<3, 3, 2, 9>,
    new tohit_calculator<AUTOMATIC_HIT>,
    nullptr,
    nullptr,
    ETC_MUTAGENIC,
    false,
    BEAM_MMISSILE,
    NUM_DCHAR_TYPES,
    false,
    true,
    TILE_BOLT_CONTAM,
},

{
    ZAP_ACID_BALL,
    "ball of acid",
    nullptr,
    nullptr,
    new dicedef_calculator<3, 5, 1, 12>,
    new tohit_calculator<40>,
    YELLOW,
    false,
    BEAM_ACID,
    DCHAR_FIRED_ZAP,
    false,
    true,
},

{
    ZAP_SHOOTING_STAR,
    "shooting star",
    new dicedef_calculator<4, 0, 1, 9>,
    new tohit_calculator<AUTOMATIC_HIT>,
    new dicedef_calculator<4, 2, 1, 14>,
    new tohit_calculator<AUTOMATIC_HIT>,
    CYAN,
    false,
    BEAM_MMISSILE,
    DCHAR_FIRED_MISSILE,
    false,
    false,
    TILE_BOLT_SEARING_RAY,
},

};
