import { Maybe, Nullable } from "./Maybe";
/**
 * Type guard to check if a value is iterable.
 * @param obj - value to check
 * @returns true if the value is iterable
 */
export declare function isIterable(obj: unknown): obj is Iterable<unknown>;
/**
 * Returns the input if it's an array, otherwise undefined.
 * @param arr - value to check
 * @returns the array if input is an array, otherwise undefined
 */
export declare function ifArray<T = unknown>(arr: T[] | unknown): Maybe<T[]>;
/**
 * Converts various input types to an array.
 * @param arr - value to convert (array, iterable, single value, or nullish)
 * @returns an array containing the input elements
 */
export declare function toArray<T>(arr: undefined | null | T[] | T | Iterable<T>): T[];
/**
 * Removes null and undefined values from an array.
 * @param array - array potentially containing nullish values
 * @returns a new array with nullish values removed
 */
export declare function compact<T>(array: Nullable<T>[]): T[];
/**
 * Remove all elements from the given array that return false from the given
 * predicate `filter`. Mutates the original array.
 * @param arr - the array to filter in place
 * @param filter - predicate function returning true for elements to keep
 * @returns the same array with non-matching elements removed
 */
export declare function filterInPlace<T>(arr: T[], filter: (t: T) => boolean): T[];
/**
 * Returns a new array with duplicate values removed (preserves first occurrence).
 * @param arr - the array to deduplicate
 * @returns a new array with unique values
 */
export declare function uniq<T>(arr: T[]): T[];
/**
 * Compares two arrays for shallow equality (same length and === elements).
 * @param a - first array
 * @param b - second array
 * @returns true if arrays have same length and identical elements by reference
 */
export declare function shallowArrayEql(a: unknown[], b: unknown[]): boolean;
type Comparable = number | string | boolean;
/**
 * Returns a copy of arr, stable sorted by the given constraint. Note that false
 * < true, and that `f` may return an array for sort priorities, or undefined if
 * the item should be skipped from the returned result.
 *
 * Note: localeSort() thinks lower case should come before upper case (!!)
 */
export declare function sortBy<T>(arr: Iterable<Maybe<T>> | Maybe<T>[], f: (t: T) => Maybe<Comparable>): T[];
/**
 * Returns the element with the minimum comparable value.
 * @param haystack - the array to search
 * @param f - function to extract a comparable value from each element
 * @returns the element with the minimum value, or undefined if array is empty
 */
export declare function leastBy<T>(haystack: T[], f: (t: T) => Maybe<Comparable>): Maybe<T>;
export {};
