"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MaintenanceWorkerService = void 0;
const common_1 = require("@nestjs/common");
const cookie_1 = require("cookie");
const jose_1 = require("jose");
const node_fs_1 = require("node:fs");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const server_dto_1 = require("../dtos/server.dto");
const enum_1 = require("../enum");
const maintenance_health_repository_1 = require("./maintenance-health.repository");
const maintenance_websocket_repository_1 = require("./maintenance-websocket.repository");
const app_repository_1 = require("../repositories/app.repository");
const config_repository_1 = require("../repositories/config.repository");
const database_repository_1 = require("../repositories/database.repository");
const logging_repository_1 = require("../repositories/logging.repository");
const process_repository_1 = require("../repositories/process.repository");
const storage_repository_1 = require("../repositories/storage.repository");
const system_metadata_repository_1 = require("../repositories/system-metadata.repository");
const database_backup_service_1 = require("../services/database-backup.service");
const config_1 = require("../utils/config");
const maintenance_1 = require("../utils/maintenance");
const misc_1 = require("../utils/misc");
let MaintenanceWorkerService = class MaintenanceWorkerService {
    logger;
    appRepository;
    configRepository;
    systemMetadataRepository;
    maintenanceWebsocketRepository;
    maintenanceHealthRepository;
    storageRepository;
    processRepository;
    databaseRepository;
    databaseBackupService;
    #secret = null;
    #status = {
        active: true,
        action: enum_1.MaintenanceAction.Start,
    };
    constructor(logger, appRepository, configRepository, systemMetadataRepository, maintenanceWebsocketRepository, maintenanceHealthRepository, storageRepository, processRepository, databaseRepository, databaseBackupService) {
        this.logger = logger;
        this.appRepository = appRepository;
        this.configRepository = configRepository;
        this.systemMetadataRepository = systemMetadataRepository;
        this.maintenanceWebsocketRepository = maintenanceWebsocketRepository;
        this.maintenanceHealthRepository = maintenanceHealthRepository;
        this.storageRepository = storageRepository;
        this.processRepository = processRepository;
        this.databaseRepository = databaseRepository;
        this.databaseBackupService = databaseBackupService;
        this.logger.setContext(this.constructor.name);
    }
    mock(status) {
        this.#secret = 'secret';
        this.#status = status;
    }
    async init() {
        const state = (await this.systemMetadataRepository.get(enum_1.SystemMetadataKey.MaintenanceMode));
        this.#secret = state.secret;
        this.#status = {
            active: true,
            action: state.action?.action ?? enum_1.MaintenanceAction.Start,
        };
        storage_core_1.StorageCore.setMediaLocation(this.detectMediaLocation());
        this.maintenanceWebsocketRepository.setAuthFn(async (client) => this.authenticate(client.request.headers));
        this.maintenanceWebsocketRepository.setStatusUpdateFn((status) => (this.#status = status));
        await this.logSecret();
        if (state.action) {
            void this.runAction(state.action);
        }
    }
    get configRepos() {
        return {
            configRepo: this.configRepository,
            metadataRepo: this.systemMetadataRepository,
            logger: this.logger,
        };
    }
    getConfig(options) {
        return (0, config_1.getConfig)(this.configRepos, options);
    }
    getSystemConfig() {
        return {
            maintenanceMode: true,
        };
    }
    getVersion() {
        return server_dto_1.ServerVersionResponseDto.fromSemVer(constants_1.serverVersion);
    }
    ssr(excludePaths) {
        const { resourcePaths } = this.configRepository.getEnv();
        let index = '';
        try {
            index = (0, node_fs_1.readFileSync)(resourcePaths.web.indexHtml).toString();
        }
        catch {
            this.logger.warn(`Unable to open ${resourcePaths.web.indexHtml}, skipping SSR.`);
        }
        return (request, res, next) => {
            if (request.url.startsWith('/api') ||
                request.method.toLowerCase() !== 'get' ||
                excludePaths.some((item) => request.url.startsWith(item))) {
                return next();
            }
            const maintenancePath = '/maintenance';
            if (!request.url.startsWith(maintenancePath)) {
                const params = new URLSearchParams();
                params.set('continue', request.path);
                return res.redirect(`${maintenancePath}?${params}`);
            }
            res.status(200).type('text/html').header('Cache-Control', 'no-store').send(index);
        };
    }
    detectMediaLocation() {
        const envData = this.configRepository.getEnv();
        if (envData.storage.mediaLocation) {
            return envData.storage.mediaLocation;
        }
        const targets = [];
        const candidates = ['/data', '/usr/src/app/upload'];
        for (const candidate of candidates) {
            const exists = this.storageRepository.existsSync(candidate);
            if (exists) {
                targets.push(candidate);
            }
        }
        if (targets.length === 1) {
            return targets[0];
        }
        return '/usr/src/app/upload';
    }
    get secret() {
        if (!this.#secret) {
            throw new Error('Secret is not initialised yet.');
        }
        return this.#secret;
    }
    get backupRepos() {
        return {
            logger: this.logger,
            storage: this.storageRepository,
            config: this.configRepository,
            process: this.processRepository,
            database: this.databaseRepository,
            health: this.maintenanceHealthRepository,
        };
    }
    getStatus() {
        return this.#status;
    }
    getPublicStatus() {
        const state = structuredClone(this.#status);
        if (state.error) {
            state.error = 'Something went wrong, see logs!';
        }
        return state;
    }
    setStatus(status) {
        this.#status = status;
        this.maintenanceWebsocketRepository.serverSend('MaintenanceStatus', status);
        this.maintenanceWebsocketRepository.clientSend('MaintenanceStatusV1', 'private', status);
        this.maintenanceWebsocketRepository.clientSend('MaintenanceStatusV1', 'public', this.getPublicStatus());
    }
    async logSecret() {
        const { server } = await this.getConfig({ withCache: true });
        const baseUrl = (0, misc_1.getExternalDomain)(server);
        const url = await (0, maintenance_1.createMaintenanceLoginUrl)(baseUrl, {
            username: 'immich-admin',
        }, this.secret);
        this.logger.log(`\n\n🚧 Immich is in maintenance mode, you can log in using the following URL:\n${url}\n`);
    }
    async authenticate(headers) {
        const jwtToken = (0, cookie_1.parse)(headers.cookie || '')[enum_1.ImmichCookie.MaintenanceToken];
        return this.login(jwtToken);
    }
    async status(potentiallyJwt) {
        try {
            await this.login(potentiallyJwt);
            return this.getStatus();
        }
        catch {
            return this.getPublicStatus();
        }
    }
    detectPriorInstall() {
        return (0, maintenance_1.detectPriorInstall)(this.storageRepository);
    }
    async login(jwt) {
        if (!jwt) {
            throw new common_1.UnauthorizedException('Missing JWT Token');
        }
        try {
            const result = await (0, jose_1.jwtVerify)(jwt, new TextEncoder().encode(this.secret));
            return result.payload;
        }
        catch {
            throw new common_1.UnauthorizedException('Invalid JWT Token');
        }
    }
    async setAction(action) {
        this.setStatus({
            active: true,
            action: action.action,
        });
        await this.runAction(action);
    }
    async runAction(action) {
        switch (action.action) {
            case enum_1.MaintenanceAction.Start: {
                return;
            }
            case enum_1.MaintenanceAction.End: {
                return this.endMaintenance();
            }
            case enum_1.MaintenanceAction.SelectDatabaseRestore: {
                return;
            }
        }
        const lock = await this.databaseRepository.tryLock(enum_1.DatabaseLock.MaintenanceOperation);
        if (!lock) {
            return;
        }
        this.logger.log(`Running maintenance action ${action.action}`);
        await this.systemMetadataRepository.set(enum_1.SystemMetadataKey.MaintenanceMode, {
            isMaintenanceMode: true,
            secret: this.secret,
            action: {
                action: enum_1.MaintenanceAction.Start,
            },
        });
        try {
            if (!action.restoreBackupFilename) {
                throw new Error("Expected restoreBackupFilename but it's missing!");
            }
            await this.restoreBackup(action.restoreBackupFilename);
        }
        catch (error) {
            this.logger.error(`Encountered error running action: ${error}`);
            this.setStatus({
                active: true,
                action: action.action,
                task: 'error',
                error: '' + error,
            });
        }
    }
    async restoreBackup(filename) {
        this.setStatus({
            active: true,
            action: enum_1.MaintenanceAction.RestoreDatabase,
            task: 'ready',
            progress: 0,
        });
        await this.databaseBackupService.restoreDatabaseBackup(filename, (task, progress) => this.setStatus({
            active: true,
            action: enum_1.MaintenanceAction.RestoreDatabase,
            progress,
            task,
        }));
        await this.setAction({
            action: enum_1.MaintenanceAction.End,
        });
    }
    async endMaintenance() {
        const state = { isMaintenanceMode: false };
        await this.systemMetadataRepository.set(enum_1.SystemMetadataKey.MaintenanceMode, state);
        this.maintenanceWebsocketRepository.clientBroadcast('AppRestartV1', state);
        this.maintenanceWebsocketRepository.serverSend('AppRestart', state);
        this.appRepository.exitApp();
    }
};
exports.MaintenanceWorkerService = MaintenanceWorkerService;
exports.MaintenanceWorkerService = MaintenanceWorkerService = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [logging_repository_1.LoggingRepository,
        app_repository_1.AppRepository,
        config_repository_1.ConfigRepository,
        system_metadata_repository_1.SystemMetadataRepository,
        maintenance_websocket_repository_1.MaintenanceWebsocketRepository,
        maintenance_health_repository_1.MaintenanceHealthRepository,
        storage_repository_1.StorageRepository,
        process_repository_1.ProcessRepository,
        database_repository_1.DatabaseRepository,
        database_backup_service_1.DatabaseBackupService])
], MaintenanceWorkerService);
//# sourceMappingURL=maintenance-worker.service.js.map