"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isObject = isObject;
exports.keys = keys;
exports.isFunction = isFunction;
exports.fromEntries = fromEntries;
exports.omit = omit;
exports.keysOf = keysOf;
/**
 * Type guard that checks if a value is a non-null, non-array object.
 *
 * @param obj - The value to check
 * @returns `true` if `obj` is a plain object (not null, not an array)
 */
function isObject(obj) {
    return obj != null && typeof obj === "object" && !Array.isArray(obj);
}
/**
 * Returns an array of own enumerable string keys from an object.
 *
 * Unlike `Object.keys`, this filters to only own enumerable properties and
 * returns an empty array for null/undefined inputs.
 *
 * @param o - The object to extract keys from
 * @returns Array of own enumerable string keys, or empty array if `o` is nullish
 */
function keys(o) {
    return o == null
        ? []
        : Object.keys(o).filter((ea) => ({}).propertyIsEnumerable.call(o, ea));
}
/**
 * Type guard that checks if a value is a function.
 *
 * @param obj - The value to check
 * @returns `true` if `obj` is a function
 */
function isFunction(obj) {
    return typeof obj === "function";
}
/**
 * Turns an array of `[key, value]` pairs into an object.
 *
 *  - Pairs whose key is `null | undefined` **or** value is `undefined` are skipped.
 *  - If `base` is provided it is mutated and returned (handy for “extend” use‑cases).
 */
function fromEntries(pairs, base = {}) {
    // don't use Object.create(null), json stringify will break!
    if (pairs == null || pairs.length === 0)
        return base ?? {};
    for (const pair of pairs) {
        if (pair?.[0] != null && pair[1] !== undefined) {
            base[pair[0]] = pair[1];
        }
    }
    return base;
}
/**
 * Returns a shallow copy of an object with the specified keys omitted.
 *
 * @param t - The source object
 * @param keysToOmit - Keys to exclude from the result
 * @returns A new object without the specified keys
 */
function omit(t, ...keysToOmit) {
    const result = {};
    for (const k of keys(t).filter((ea) => !keysToOmit.includes(ea))) {
        result[k] = t[k];
    }
    return result;
}
/**
 * Provides a type-safe exhaustive array of keys for a given interface.
 *
 * Unfortunately, `satisfies (keyof T)[]` doesn't ensure all keys are present,
 * and doesn't guard against duplicates. This function does.
 *
 * @param t - The interface to extract keys from. This is a Record of keys to
 * `true`, which ensures the returned key array is unique.
 */
function keysOf(t) {
    return Object.keys(t);
}
// This also doesn't enforce that all keys are present:
// type RequiredKeys<T> = { [K in keyof Required<T>]: K } extends { [K: string]: infer U } ? U[] : never;
//# sourceMappingURL=Object.js.map