"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isPlatformCaseSensitive = void 0;
exports.isFileEmpty = isFileEmpty;
exports.compareFilePaths = compareFilePaths;
const promises_1 = require("node:fs/promises");
const node_path_1 = require("node:path");
const Lazy_1 = require("./Lazy");
const String_1 = require("./String");
/**
 * Checks if a file is empty or does not exist.
 * @param path - the file path to check
 * @returns true if the file is empty or does not exist, false otherwise
 * @throws if path is blank or if a non-ENOENT error occurs
 */
async function isFileEmpty(path) {
    if ((0, String_1.blank)(path)) {
        throw new Error("isFileEmpty(): blank path");
    }
    try {
        const s = await (0, promises_1.stat)(path);
        return s == null || s.size === 0;
    }
    catch (err) {
        if (err &&
            typeof err === "object" &&
            "code" in err &&
            err.code === "ENOENT")
            return true;
        else
            throw err;
    }
}
/**
 * Returns true if the current platform has case-sensitive file paths.
 */
exports.isPlatformCaseSensitive = (0, Lazy_1.lazy)(() => process.platform !== "win32" && process.platform !== "darwin");
/**
 * Compares two file paths for equality, respecting platform case sensitivity.
 * @param a - first file path
 * @param b - second file path
 * @returns true if paths refer to the same file
 */
function compareFilePaths(a, b) {
    const aNorm = (0, node_path_1.normalize)(a);
    const bNorm = (0, node_path_1.normalize)(b);
    return (0, exports.isPlatformCaseSensitive)()
        ? aNorm === bNorm
        : aNorm.localeCompare(bNorm, undefined, { sensitivity: "base" }) === 0;
}
//# sourceMappingURL=File.js.map