"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BinaryExtractionTask = void 0;
const node_fs_1 = require("node:fs");
const node_path_1 = __importDefault(require("node:path"));
const ExifToolTask_1 = require("./ExifToolTask");
const FilenameCharsetArgs_1 = require("./FilenameCharsetArgs");
const String_1 = require("./String");
const StdoutRe = /\b(\d+) output files? created/i;
/**
 * Task that returns an error string (to prevent retries), or undefined if
 * everything seems to have worked.
 */
class BinaryExtractionTask extends ExifToolTask_1.ExifToolTask {
    constructor(args, options) {
        super(args, options);
    }
    /**
     * Creates a task to extract a binary tag value to a file.
     * @param tagname the tag containing the binary data (e.g., "ThumbnailImage", "PreviewImage")
     * @param imgSrc the source image file path
     * @param imgDest the destination file path for the extracted binary data
     * @param options extraction options
     * @returns a new BinaryExtractionTask instance
     */
    static for(tagname, imgSrc, imgDest, options) {
        // Ensure the destination directory exists:
        (0, node_fs_1.mkdirSync)(node_path_1.default.dirname(imgDest), { recursive: true });
        const forceWrite = options?.forceWrite ?? false;
        const args = [
            ...FilenameCharsetArgs_1.Utf8FilenameCharsetArgs,
            "-b", // -binary
            "-" + tagname,
            // Capital W allows destination files to not have a pattern. See
            // https://exiftool.org/exiftool_pod.html#W-FMT--tagOut
            //
            // Prior code used -w with %0f "to avoid shell expansion", but as this
            // command gets sent directly to exiftool thanks to stay_open mode, we
            // don't need to worry about shell expansion.
            //
            // I also tried `-out` instead of `-W`, and that didn't work at least on
            // ExifTool 13.17.
            forceWrite ? "-W!" : "-W",
            node_path_1.default.resolve(imgDest),
            node_path_1.default.resolve(imgSrc),
        ];
        return new BinaryExtractionTask(args, options);
    }
    parse(stdout, err) {
        const s = (0, String_1.toS)(stdout).trim();
        const m = StdoutRe.exec(s);
        if (err != null) {
            throw err;
        }
        else if (m == null) {
            throw new Error("Missing expected status message (got " + stdout + ")");
        }
        else if (m[1] === "1") {
            return;
        }
        else {
            // Don't retry: the binary payload is missing, and retrying won't fix that.
            return s;
        }
    }
}
exports.BinaryExtractionTask = BinaryExtractionTask;
//# sourceMappingURL=BinaryExtractionTask.js.map