"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProcessPoolManager = void 0;
const node_timers_1 = __importDefault(require("node:timers"));
const Array_1 = require("./Array");
const BatchProcess_1 = require("./BatchProcess");
const Error_1 = require("./Error");
const ProcessHealthMonitor_1 = require("./ProcessHealthMonitor");
const Timeout_1 = require("./Timeout");
/**
 * Manages the lifecycle of a pool of BatchProcess instances.
 * Handles spawning, health monitoring, and cleanup of child processes.
 */
class ProcessPoolManager {
    options;
    emitter;
    onIdle;
    #procs = [];
    #logger;
    #healthMonitor;
    #nextSpawnTime = 0;
    #lastPidsCheckTime = 0;
    #spawnedProcs = 0;
    constructor(options, emitter, onIdle) {
        this.options = options;
        this.emitter = emitter;
        this.onIdle = onIdle;
        this.#logger = options.logger;
        this.#healthMonitor = new ProcessHealthMonitor_1.ProcessHealthMonitor(options, emitter);
    }
    /**
     * Get all current processes
     */
    get processes() {
        return this.#procs;
    }
    /**
     * Get the current number of spawned child processes
     */
    get procCount() {
        return this.#procs.length;
    }
    /**
     * Alias for procCount to match BatchCluster interface
     */
    get processCount() {
        return this.procCount;
    }
    /**
     * Get the current number of child processes currently servicing tasks
     */
    get busyProcCount() {
        return (0, Array_1.count)(this.#procs, 
        // don't count procs that are starting up as "busy":
        (ea) => !ea.starting && !ea.ending && !ea.idle);
    }
    /**
     * Get the current number of starting processes
     */
    get startingProcCount() {
        return (0, Array_1.count)(this.#procs, 
        // don't count procs that are starting up as "busy":
        (ea) => ea.starting && !ea.ending);
    }
    /**
     * Get the current number of ready processes
     */
    get readyProcCount() {
        return (0, Array_1.count)(this.#procs, (ea) => ea.ready);
    }
    /**
     * Get the total number of child processes created by this instance
     */
    get spawnedProcCount() {
        return this.#spawnedProcs;
    }
    /**
     * Get the milliseconds until the next spawn is allowed
     */
    get msBeforeNextSpawn() {
        return Math.max(0, this.#nextSpawnTime - Date.now());
    }
    /**
     * Get all currently running tasks from all processes
     */
    currentTasks() {
        const tasks = [];
        for (const proc of this.#procs) {
            if (proc.currentTask != null) {
                tasks.push(proc.currentTask);
            }
        }
        return tasks;
    }
    /**
     * Find the first ready process that can handle a new task
     */
    findReadyProcess() {
        return this.#procs.find((ea) => ea.ready);
    }
    /**
     * Verify that each BatchProcess PID is actually alive.
     * @return the spawned PIDs that are still in the process table.
     */
    pids() {
        const arr = [];
        for (const proc of [...this.#procs]) {
            if (proc != null && proc.running()) {
                arr.push(proc.pid);
            }
        }
        return arr;
    }
    /**
     * Shut down any currently-running child processes.
     */
    async closeChildProcesses(gracefully = true) {
        const procs = [...this.#procs];
        this.#procs.length = 0;
        await Promise.all(procs.map((proc) => proc
            .end(gracefully, "ending")
            .catch((err) => this.emitter.emit("endError", (0, Error_1.asError)(err), proc))));
    }
    /**
     * Run maintenance on currently spawned child processes.
     * Removes unhealthy processes and enforces maxProcs limit.
     */
    vacuumProcs() {
        this.#maybeCheckPids();
        const endPromises = [];
        let pidsToReap = Math.max(0, this.#procs.length - this.options.maxProcs);
        (0, Array_1.filterInPlace)(this.#procs, (proc) => {
            // Only check `.idle` (not `.ready`) procs. We don't want to reap busy
            // procs unless we're ending, and unhealthy procs (that we want to reap)
            // won't be `.ready`.
            if (proc.idle) {
                // don't reap more than pidsToReap pids. We can't use #procs.length
                // within filterInPlace because #procs.length only changes at iteration
                // completion: the prior impl resulted in all idle pids getting reaped
                // when maxProcs was reduced.
                const why = proc.whyNotHealthy ?? (--pidsToReap >= 0 ? "tooMany" : null);
                if (why != null) {
                    endPromises.push(proc.end(true, why));
                    return false;
                }
                proc.maybeRunHealthCheck();
            }
            return true;
        });
        return Promise.all(endPromises);
    }
    /**
     * Spawn new processes if needed based on pending task count and capacity
     */
    async maybeSpawnProcs(pendingTaskCount, ended) {
        let procsToSpawn = this.#procsToSpawn(pendingTaskCount);
        if (ended || this.#nextSpawnTime > Date.now() || procsToSpawn === 0) {
            return;
        }
        // prevent concurrent runs:
        this.#nextSpawnTime = Date.now() + this.#maxSpawnDelay();
        for (let i = 0; i < procsToSpawn; i++) {
            if (ended) {
                break;
            }
            // Kick the lock down the road:
            this.#nextSpawnTime = Date.now() + this.#maxSpawnDelay();
            this.#spawnedProcs++;
            try {
                const proc = this.#spawnNewProc();
                const result = await (0, Timeout_1.thenOrTimeout)(proc, this.options.spawnTimeoutMillis);
                if (result === Timeout_1.Timeout) {
                    void proc
                        .then((bp) => {
                        void bp.end(false, "startError");
                        this.emitter.emit("startError", (0, Error_1.asError)("Failed to spawn process in " +
                            this.options.spawnTimeoutMillis +
                            "ms"), bp);
                    })
                        .catch((err) => {
                        // this should only happen if the processFactory throws a
                        // rejection:
                        this.emitter.emit("startError", (0, Error_1.asError)(err));
                    });
                }
                else {
                    this.#logger().debug("ProcessPoolManager.maybeSpawnProcs() started healthy child process", { pid: result.pid });
                }
                // tasks may have been popped off or setMaxProcs may have reduced
                // maxProcs. Do this at the end so the for loop ends properly.
                procsToSpawn = Math.min(this.#procsToSpawn(pendingTaskCount), procsToSpawn);
            }
            catch (err) {
                this.emitter.emit("startError", (0, Error_1.asError)(err));
            }
        }
        // YAY WE MADE IT.
        // Only let more children get spawned after minDelay:
        const delay = Math.max(100, this.options.minDelayBetweenSpawnMillis);
        this.#nextSpawnTime = Date.now() + delay;
        // And schedule #onIdle for that time:
        node_timers_1.default.setTimeout(this.onIdle, delay).unref();
    }
    /**
     * Update the maximum number of processes allowed
     */
    setMaxProcs(maxProcs) {
        this.options.maxProcs = maxProcs;
    }
    #maybeCheckPids() {
        if (this.options.cleanupChildProcs &&
            this.options.pidCheckIntervalMillis > 0 &&
            this.#lastPidsCheckTime + this.options.pidCheckIntervalMillis < Date.now()) {
            this.#lastPidsCheckTime = Date.now();
            void this.pids();
        }
    }
    #maxSpawnDelay() {
        // 10s delay is certainly long enough for .spawn() to return, even on a
        // loaded windows machine.
        return Math.max(10_000, this.options.spawnTimeoutMillis);
    }
    #procsToSpawn(pendingTaskCount) {
        const remainingCapacity = this.options.maxProcs - this.#procs.length;
        // take into account starting procs, so one task doesn't result in multiple
        // processes being spawned:
        const requestedCapacity = pendingTaskCount - this.startingProcCount;
        const atLeast0 = Math.max(0, Math.min(remainingCapacity, requestedCapacity));
        return this.options.minDelayBetweenSpawnMillis === 0
            ? // we can spin up multiple processes in parallel.
                atLeast0
            : // Don't spin up more than 1:
                Math.min(1, atLeast0);
    }
    // must only be called by this.maybeSpawnProcs()
    async #spawnNewProc() {
        // no matter how long it takes to spawn, always push the result into #procs
        // so we don't leak child processes:
        const procOrPromise = this.options.processFactory();
        const proc = await procOrPromise;
        const result = new BatchProcess_1.BatchProcess(proc, this.options, this.onIdle, this.#healthMonitor);
        this.#procs.push(result);
        return result;
    }
}
exports.ProcessPoolManager = ProcessPoolManager;
//# sourceMappingURL=ProcessPoolManager.js.map