"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readConstraints = void 0;
const kysely_1 = require("kysely");
const types_1 = require("../types");
const readConstraints = async (ctx, db) => {
    const constraints = await db
        .selectFrom('pg_constraint')
        .innerJoin('pg_namespace', 'pg_namespace.oid', 'pg_constraint.connamespace')
        .innerJoin('pg_class as source_table', (join) => join.onRef('source_table.oid', '=', 'pg_constraint.conrelid').on('source_table.relkind', 'in', [
        kysely_1.sql.lit('r'),
        kysely_1.sql.lit('p'),
        kysely_1.sql.lit('f'),
    ]))
        .leftJoin('pg_class as reference_table', 'reference_table.oid', 'pg_constraint.confrelid')
        .select((eb) => [
        'pg_constraint.contype as constraint_type',
        'pg_constraint.conname as constraint_name',
        'source_table.relname as table_name',
        'reference_table.relname as reference_table_name',
        'pg_constraint.confupdtype as update_action',
        'pg_constraint.confdeltype as delete_action',
        eb
            .selectFrom('pg_attribute')
            .whereRef('pg_attribute.attrelid', '=', 'pg_constraint.conrelid')
            .whereRef('pg_attribute.attnum', '=', (0, kysely_1.sql) `any("pg_constraint"."conkey")`)
            .select((eb) => eb.fn('json_agg', ['pg_attribute.attname']).as('column_name'))
            .as('column_names'),
        eb
            .selectFrom('pg_attribute')
            .whereRef('pg_attribute.attrelid', '=', 'pg_constraint.confrelid')
            .whereRef('pg_attribute.attnum', '=', (0, kysely_1.sql) `any("pg_constraint"."confkey")`)
            .select((eb) => eb.fn('json_agg', ['pg_attribute.attname']).as('column_name'))
            .as('reference_column_names'),
        eb.fn('pg_get_constraintdef', ['pg_constraint.oid']).as('expression'),
    ])
        .where('pg_namespace.nspname', '=', ctx.schemaName)
        .execute();
    for (const constraint of constraints) {
        const table = ctx.getTableByName(constraint.table_name);
        if (!table) {
            continue;
        }
        const constraintName = constraint.constraint_name;
        switch (constraint.constraint_type) {
            case 'p': {
                if (!constraint.column_names) {
                    ctx.warnings.push(`Skipping CONSTRAINT "${constraintName}", no columns found`);
                    continue;
                }
                table.constraints.push({
                    type: types_1.ConstraintType.PRIMARY_KEY,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    synchronize: true,
                });
                break;
            }
            case 'f': {
                if (!constraint.column_names || !constraint.reference_table_name || !constraint.reference_column_names) {
                    ctx.warnings.push(`Skipping CONSTRAINT "${constraintName}", missing either columns, referenced table, or referenced columns,`);
                    continue;
                }
                table.constraints.push({
                    type: types_1.ConstraintType.FOREIGN_KEY,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    referenceTableName: constraint.reference_table_name,
                    referenceColumnNames: constraint.reference_column_names,
                    onUpdate: asDatabaseAction(constraint.update_action),
                    onDelete: asDatabaseAction(constraint.delete_action),
                    synchronize: true,
                });
                break;
            }
            case 'u': {
                table.constraints.push({
                    type: types_1.ConstraintType.UNIQUE,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    synchronize: true,
                });
                break;
            }
            case 'c': {
                table.constraints.push({
                    type: types_1.ConstraintType.CHECK,
                    name: constraint.constraint_name,
                    tableName: constraint.table_name,
                    expression: constraint.expression.replace('CHECK ', ''),
                    synchronize: true,
                });
                break;
            }
        }
    }
};
exports.readConstraints = readConstraints;
const asDatabaseAction = (action) => {
    switch (action) {
        case 'a': {
            return types_1.ActionType.NO_ACTION;
        }
        case 'c': {
            return types_1.ActionType.CASCADE;
        }
        case 'r': {
            return types_1.ActionType.RESTRICT;
        }
        case 'n': {
            return types_1.ActionType.SET_NULL;
        }
        case 'd': {
            return types_1.ActionType.SET_DEFAULT;
        }
        default: {
            return types_1.ActionType.NO_ACTION;
        }
    }
};
//# sourceMappingURL=constraint.reader.js.map